<?php
/**
 * M3U Playlist Parser
 * Parses M3U and M3U_Plus format playlists
 */

class M3UParser
{
    private array $entries = [];
    private array $groups = [];
    private array $errors = [];

    // Regex patterns for parsing
    private const EXTINF_PATTERN = '/^#EXTINF:\s*(-?\d+)\s*(.*)$/';
    private const ATTRIBUTE_PATTERN = '/([a-zA-Z0-9_-]+)=["\']([^"\']*)["\']|([a-zA-Z0-9_-]+)=([^\s,]+)/';
    private const STREAM_TYPE_PATTERNS = [
        'live' => '/\/live\//',
        'movie' => '/\/movie\/|\/vod\//',
        'series' => '/\/series\//'
    ];

    /**
     * Parse M3U content from string
     */
    public function parseContent(string $content): array
    {
        $this->entries = [];
        $this->groups = [];
        $this->errors = [];

        $lines = preg_split('/\r\n|\r|\n/', $content);
        $totalLines = count($lines);

        $currentEntry = null;

        for ($i = 0; $i < $totalLines; $i++) {
            $line = trim($lines[$i]);

            // Skip empty lines and M3U header
            if (empty($line) || $line === '#EXTM3U') {
                continue;
            }

            // Parse EXTINF line
            if (strpos($line, '#EXTINF:') === 0) {
                $currentEntry = $this->parseExtInf($line);
                continue;
            }

            // Parse stream URL (non-comment line after EXTINF)
            if ($currentEntry !== null && strpos($line, '#') !== 0) {
                $currentEntry['url'] = $line;
                $currentEntry['stream_type'] = $this->detectStreamType($line);
                $currentEntry['stream_id'] = $this->extractStreamId($line);
                $currentEntry['container'] = $this->extractContainer($line);

                // Parse movie/series info from name
                $parsedInfo = $this->parseMediaName($currentEntry['name'] ?? '');
                $currentEntry = array_merge($currentEntry, $parsedInfo);

                // Generate clean filename
                $currentEntry['clean_name'] = $this->generateCleanName($currentEntry);

                $this->entries[] = $currentEntry;

                // Track groups/categories
                if (!empty($currentEntry['group_title'])) {
                    $this->groups[$currentEntry['group_title']] =
                        ($this->groups[$currentEntry['group_title']] ?? 0) + 1;
                }

                $currentEntry = null;
            }
        }

        return [
            'entries' => $this->entries,
            'groups' => $this->groups,
            'total' => count($this->entries),
            'errors' => $this->errors
        ];
    }

    /**
     * Parse M3U from URL
     */
    public function parseUrl(string $url, int $timeout = 60): array
    {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_USERAGENT => 'TiviMate/5.1.6 (Android 7.1.2)',
            CURLOPT_HTTPHEADER => [
                'Accept: */*',
                'Connection: keep-alive'
            ]
        ]);

        $content = curl_exec($ch);
        $error = curl_error($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($content === false || $httpCode !== 200) {
            return [
                'success' => false,
                'error' => $error ?: "HTTP Error: $httpCode",
                'entries' => [],
                'groups' => [],
                'total' => 0
            ];
        }

        $result = $this->parseContent($content);
        $result['success'] = true;
        $result['source_url'] = $url;

        return $result;
    }

    /**
     * Parse M3U from file
     */
    public function parseFile(string $filepath): array
    {
        if (!file_exists($filepath)) {
            return [
                'success' => false,
                'error' => "File not found: $filepath",
                'entries' => [],
                'groups' => [],
                'total' => 0
            ];
        }

        $content = file_get_contents($filepath);
        $result = $this->parseContent($content);
        $result['success'] = true;
        $result['source_file'] = $filepath;

        return $result;
    }

    /**
     * Parse EXTINF line and extract attributes
     */
    private function parseExtInf(string $line): array
    {
        $entry = [
            'duration' => -1,
            'name' => '',
            'tvg_id' => '',
            'tvg_name' => '',
            'tvg_logo' => '',
            'group_title' => '',
            'attributes' => []
        ];

        // Match the basic EXTINF format
        if (preg_match(self::EXTINF_PATTERN, $line, $matches)) {
            $entry['duration'] = (int) $matches[1];

            $attributeStr = $matches[2] ?? '';

            // Extract quoted and unquoted attributes
            if (preg_match_all(self::ATTRIBUTE_PATTERN, $attributeStr, $attrMatches, PREG_SET_ORDER)) {
                foreach ($attrMatches as $attr) {
                    $key = !empty($attr[1]) ? $attr[1] : $attr[3];
                    $value = !empty($attr[2]) ? $attr[2] : ($attr[4] ?? '');

                    $entry['attributes'][$key] = $value;

                    // Map common attributes
                    switch (strtolower($key)) {
                        case 'tvg-id':
                            $entry['tvg_id'] = $value;
                            break;
                        case 'tvg-name':
                            $entry['tvg_name'] = $value;
                            break;
                        case 'tvg-logo':
                            $entry['tvg_logo'] = $value;
                            break;
                        case 'group-title':
                            $entry['group_title'] = $value;
                            break;
                    }
                }
            }

            // Extract channel/movie name (after last comma)
            $lastCommaPos = strrpos($attributeStr, ',');
            if ($lastCommaPos !== false) {
                $entry['name'] = trim(substr($attributeStr, $lastCommaPos + 1));
            } elseif (empty($entry['tvg_name'])) {
                // Fallback: use everything after attributes
                $entry['name'] = trim(preg_replace('/^.*"\s*,?\s*/', '', $attributeStr));
            }

            // Use tvg-name if name is empty
            if (empty($entry['name']) && !empty($entry['tvg_name'])) {
                $entry['name'] = $entry['tvg_name'];
            }
        }

        return $entry;
    }

    /**
     * Detect stream type from URL
     */
    private function detectStreamType(string $url): string
    {
        foreach (self::STREAM_TYPE_PATTERNS as $type => $pattern) {
            if (preg_match($pattern, $url)) {
                return $type;
            }
        }
        return 'unknown';
    }

    /**
     * Extract stream ID from URL
     */
    private function extractStreamId(string $url): ?int
    {
        // Pattern: /username/password/12345.ext
        if (preg_match('/\/(\d+)\.[a-zA-Z0-9]+$/', $url, $matches)) {
            return (int) $matches[1];
        }
        return null;
    }

    /**
     * Extract container/extension from URL
     */
    private function extractContainer(string $url): string
    {
        // Remove query string
        $path = parse_url($url, PHP_URL_PATH) ?? '';

        if (preg_match('/\.([a-zA-Z0-9]+)$/', $path, $matches)) {
            return strtolower($matches[1]);
        }

        return 'ts'; // default
    }

    /**
     * Parse media name to extract title, year, season, episode
     */
    private function parseMediaName(string $name): array
    {
        $result = [
            'parsed_title' => $name,
            'parsed_year' => null,
            'parsed_season' => null,
            'parsed_episode' => null,
            'is_series' => false
        ];

        // Clean the name
        $cleanName = trim($name);

        // Try to extract year: "Movie Name (2025)" or "Movie Name 2025"
        if (preg_match('/^(.+?)\s*[\(\[]?\s*(19\d{2}|20\d{2})\s*[\)\]]?\s*$/i', $cleanName, $matches)) {
            $result['parsed_title'] = trim($matches[1]);
            $result['parsed_year'] = (int) $matches[2];
        }

        // Try to extract season/episode: S01E01, s01.e01, Season 1 Episode 1, etc.
        $seasonEpPatterns = [
            '/[Ss](\d{1,2})\s*[Ee](\d{1,3})/',           // S01E01
            '/[Ss](\d{1,2})\.?[Ee](\d{1,3})/',          // s01.e01
            '/[Ss]eason\s*(\d{1,2}).*?[Ee]pisode\s*(\d{1,3})/i', // Season 1 Episode 1
            '/(\d{1,2})[xX](\d{1,3})/',                  // 1x01
        ];

        foreach ($seasonEpPatterns as $pattern) {
            if (preg_match($pattern, $cleanName, $matches)) {
                $result['parsed_season'] = (int) $matches[1];
                $result['parsed_episode'] = (int) $matches[2];
                $result['is_series'] = true;

                // Remove season/episode info from title
                $result['parsed_title'] = trim(preg_replace($pattern, '', $result['parsed_title']));
                break;
            }
        }

        // Clean up title
        $result['parsed_title'] = $this->cleanTitle($result['parsed_title']);

        return $result;
    }

    /**
     * Clean up title string
     */
    private function cleanTitle(string $title): string
    {
        // Remove common tags
        $title = preg_replace('/\[.*?\]|\(.*?(?:HD|SD|4K|CAM|TS|HDCAM).*?\)/i', '', $title);

        // Remove quality indicators
        $title = preg_replace('/\b(720p|1080p|2160p|4K|HD|SD|CAM|TS|HDCAM|BluRay|WEB-?DL|WEBRip|HDRip|DVDRip)\b/i', '', $title);

        // Replace dots, underscores with spaces
        $title = preg_replace('/[._]+/', ' ', $title);

        // Remove extra whitespace
        $title = preg_replace('/\s+/', ' ', $title);

        return trim($title);
    }

    /**
     * Generate clean filename for FolderWatch compatibility
     */
    private function generateCleanName(array $entry): string
    {
        $title = $entry['parsed_title'] ?? $entry['name'];
        $year = $entry['parsed_year'];
        $season = $entry['parsed_season'];
        $episode = $entry['parsed_episode'];
        $extension = $entry['container'] ?? 'mp4';

        // Clean title: replace spaces with dots, remove special chars
        $cleanTitle = preg_replace('/[^\w\s\-]/u', '', $title);
        $cleanTitle = preg_replace('/\s+/', '.', trim($cleanTitle));

        // Build filename
        if ($entry['is_series'] && $season !== null && $episode !== null) {
            // Series format: Series.Name (2024).s01.e01.ext
            $name = $cleanTitle;
            if ($year) {
                $name .= " ($year)";
            }
            $name .= sprintf('.s%02d.e%02d', $season, $episode);
        } else {
            // Movie format: Movie.Name (2025).ext
            $name = $cleanTitle;
            if ($year) {
                $name .= " ($year)";
            }
        }

        return $name . '.' . $extension;
    }

    /**
     * Filter entries by type
     */
    public function filterByType(string $type): array
    {
        return array_filter($this->entries, function ($entry) use ($type) {
            return ($entry['stream_type'] ?? 'unknown') === $type;
        });
    }

    /**
     * Filter entries by group/category
     */
    public function filterByGroup(string $group): array
    {
        return array_filter($this->entries, function ($entry) use ($group) {
            return ($entry['group_title'] ?? '') === $group;
        });
    }

    /**
     * Get movies only
     */
    public function getMovies(): array
    {
        return $this->filterByType('movie');
    }

    /**
     * Get series/episodes only
     */
    public function getSeries(): array
    {
        return $this->filterByType('series');
    }

    /**
     * Get live channels only
     */
    public function getLive(): array
    {
        return $this->filterByType('live');
    }

    /**
     * Get all groups/categories
     */
    public function getGroups(): array
    {
        return $this->groups;
    }

    /**
     * Get all entries
     */
    public function getEntries(): array
    {
        return $this->entries;
    }

    /**
     * Get parsing errors
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * Validate entry name for FolderWatch compatibility
     */
    public static function isValidFolderWatchName(string $filename): bool
    {
        // Movie pattern: name (year).ext
        $moviePattern = '/^.+\s*\(\d{4}\)\.[a-zA-Z0-9]+$/';

        // Series pattern: name (year).sXX.eXX.ext
        $seriesPattern = '/^.+\s*\(\d{4}\)\.s\d{2}\.e\d{2,3}\.[a-zA-Z0-9]+$/i';

        return preg_match($moviePattern, $filename) || preg_match($seriesPattern, $filename);
    }

    /**
     * Suggest fix for invalid filename
     */
    public static function suggestValidName(string $filename): ?string
    {
        // Try to extract info and rebuild
        $info = (new self())->parseMediaName(pathinfo($filename, PATHINFO_FILENAME));
        $ext = pathinfo($filename, PATHINFO_EXTENSION) ?: 'mp4';

        if (empty($info['parsed_title'])) {
            return null;
        }

        $title = preg_replace('/\s+/', '.', trim($info['parsed_title']));

        if ($info['is_series'] && $info['parsed_season'] && $info['parsed_episode']) {
            $year = $info['parsed_year'] ?? date('Y');
            return sprintf('%s (%d).s%02d.e%02d.%s',
                $title, $year, $info['parsed_season'], $info['parsed_episode'], $ext);
        }

        if ($info['parsed_year']) {
            return sprintf('%s (%d).%s', $title, $info['parsed_year'], $ext);
        }

        return null;
    }
}
