<?php
/**
 * Remote Downloader
 * Manages downloading VOD content to remote servers via SSH/FTP/SFTP
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/Logger.php';

class RemoteDownloader
{
    private PDO $db;
    private Logger $logger;

    public function __construct()
    {
        $this->db = getDB();
        $this->logger = new Logger('remote_downloader');
    }

    /**
     * Add a remote server
     */
    public function addServer(array $data): ?int
    {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO remote_servers (name, type, host, port, username, password, private_key, remote_path, local_path, is_active, priority, config)
                VALUES (:name, :type, :host, :port, :username, :password, :private_key, :remote_path, :local_path, :is_active, :priority, :config)
            ");

            // Encrypt password if provided
            $password = isset($data['password']) ? $this->encryptPassword($data['password']) : null;
            $privateKey = isset($data['private_key']) ? $this->encryptPassword($data['private_key']) : null;

            $stmt->execute([
                'name' => $data['name'],
                'type' => $data['type'] ?? 'ssh',
                'host' => $data['host'],
                'port' => $data['port'] ?? ($data['type'] === 'ftp' ? 21 : 22),
                'username' => $data['username'],
                'password' => $password,
                'private_key' => $privateKey,
                'remote_path' => $data['remote_path'] ?? '/var/www/html/media',
                'local_path' => $data['local_path'] ?? '/var/www/html/VOD/media',
                'is_active' => $data['is_active'] ?? 1,
                'priority' => $data['priority'] ?? 1,
                'config' => isset($data['config']) ? json_encode($data['config']) : null
            ]);

            $serverId = (int) $this->db->lastInsertId();
            $this->logger->info("Remote server added", ['server_id' => $serverId, 'name' => $data['name']]);

            return $serverId;

        } catch (PDOException $e) {
            $this->logger->error("Failed to add remote server", ['error' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Update remote server
     */
    public function updateServer(int $id, array $data): bool
    {
        try {
            $fields = [];
            $params = ['id' => $id];

            $allowedFields = ['name', 'type', 'host', 'port', 'username', 'remote_path', 'local_path', 'is_active', 'priority', 'status', 'error_message', 'last_check'];

            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $fields[] = "{$field} = :{$field}";
                    $params[$field] = $data[$field];
                }
            }

            if (isset($data['password']) && !empty($data['password'])) {
                $fields[] = "password = :password";
                $params['password'] = $this->encryptPassword($data['password']);
            }

            if (isset($data['private_key']) && !empty($data['private_key'])) {
                $fields[] = "private_key = :private_key";
                $params['private_key'] = $this->encryptPassword($data['private_key']);
            }

            if (isset($data['config'])) {
                $fields[] = "config = :config";
                $params['config'] = json_encode($data['config']);
            }

            if (empty($fields)) {
                return false;
            }

            $sql = "UPDATE remote_servers SET " . implode(', ', $fields) . " WHERE id = :id";
            $stmt = $this->db->prepare($sql);
            return $stmt->execute($params);

        } catch (PDOException $e) {
            $this->logger->error("Failed to update remote server", ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Delete remote server
     */
    public function deleteServer(int $id): bool
    {
        try {
            $stmt = $this->db->prepare("DELETE FROM remote_servers WHERE id = ?");
            return $stmt->execute([$id]);
        } catch (PDOException $e) {
            $this->logger->error("Failed to delete remote server", ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Get remote server by ID
     */
    public function getServer(int $id): ?array
    {
        $stmt = $this->db->prepare("SELECT * FROM remote_servers WHERE id = ?");
        $stmt->execute([$id]);
        $server = $stmt->fetch();

        if ($server) {
            if ($server['config']) {
                $server['config'] = json_decode($server['config'], true);
            }
            // Don't decrypt password in get - only when needed
        }

        return $server ?: null;
    }

    /**
     * Get all remote servers
     */
    public function getAllServers(bool $activeOnly = false): array
    {
        $sql = "SELECT id, name, type, host, port, username, remote_path, local_path, is_active, priority, status, last_check, error_message, created_at FROM remote_servers";
        if ($activeOnly) {
            $sql .= " WHERE is_active = 1";
        }
        $sql .= " ORDER BY priority DESC, name ASC";

        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }

    /**
     * Test connection to remote server
     */
    public function testConnection(int $serverId): array
    {
        $server = $this->getServer($serverId);
        if (!$server) {
            return ['success' => false, 'error' => 'Server not found'];
        }

        $result = ['success' => false, 'message' => '', 'response_time' => 0];
        $startTime = microtime(true);

        try {
            switch ($server['type']) {
                case 'ssh':
                case 'sftp':
                    $result = $this->testSSHConnection($server);
                    break;
                case 'ftp':
                    $result = $this->testFTPConnection($server);
                    break;
                default:
                    $result = ['success' => false, 'error' => 'Unknown server type'];
            }

            $result['response_time'] = round((microtime(true) - $startTime) * 1000);

            // Update server status
            $this->updateServer($serverId, [
                'status' => $result['success'] ? 'online' : 'error',
                'last_check' => date('Y-m-d H:i:s'),
                'error_message' => $result['error'] ?? null
            ]);

        } catch (Exception $e) {
            $result = [
                'success' => false,
                'error' => $e->getMessage(),
                'response_time' => round((microtime(true) - $startTime) * 1000)
            ];

            $this->updateServer($serverId, [
                'status' => 'error',
                'last_check' => date('Y-m-d H:i:s'),
                'error_message' => $e->getMessage()
            ]);
        }

        $this->logger->info("Connection test", array_merge(['server_id' => $serverId], $result));

        return $result;
    }

    /**
     * Get storage info from remote server
     */
    public function getServerStorage(int $serverId): array
    {
        $server = $this->getServer($serverId);
        if (!$server) {
            return ['success' => false, 'error' => 'Server not found'];
        }

        try {
            $password = $this->decryptPassword($server['password']);
            $path = $server['remote_path'] ?: '/';

            if ($server['type'] === 'ftp') {
                return ['success' => false, 'error' => 'Storage check not supported for FTP'];
            }

            // Use sshpass for SSH/SFTP
            $port = $server['port'] ?: 22;
            $dfCommand = "df -B1 " . escapeshellarg($path) . " 2>/dev/null | tail -1";

            $cmd = sprintf(
                'sshpass -p %s ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 -p %d %s@%s %s 2>&1',
                escapeshellarg($password),
                $port,
                escapeshellarg($server['username']),
                escapeshellarg($server['host']),
                escapeshellarg($dfCommand)
            );

            $output = shell_exec($cmd);

            if (empty($output)) {
                return ['success' => false, 'error' => 'Failed to get storage info - empty output'];
            }

            // Parse df output: Filesystem 1B-blocks Used Available Use% Mounted
            // Find the line that starts with /dev (the actual df data)
            $lines = explode("\n", trim($output));
            $dfLine = null;
            foreach ($lines as $line) {
                if (preg_match('/^\/dev/', trim($line)) || preg_match('/^\d+\s/', trim($line))) {
                    $dfLine = trim($line);
                    break;
                }
            }

            if (!$dfLine) {
                // Try last line as fallback
                $dfLine = trim(end($lines));
            }

            $parts = preg_split('/\s+/', $dfLine);

            // df -B1 output: [0]=filesystem, [1]=total, [2]=used, [3]=available, [4]=percent, [5]=mount
            if (count($parts) >= 5 && is_numeric($parts[1])) {
                $total = (float) $parts[1];
                $used = (float) $parts[2];
                $free = (float) $parts[3];
                $percent = $total > 0 ? round(($used / $total) * 100, 1) : 0;

                return [
                    'success' => true,
                    'total' => $total,
                    'used' => $used,
                    'free' => $free,
                    'percent_used' => $percent,
                    'total_formatted' => $this->formatBytes($total),
                    'used_formatted' => $this->formatBytes($used),
                    'free_formatted' => $this->formatBytes($free)
                ];
            }

            // Check for actual connection errors
            if (strpos($output, 'Connection refused') !== false || strpos($output, 'No route to host') !== false) {
                return ['success' => false, 'error' => 'Connection failed'];
            }

            return ['success' => false, 'error' => 'Failed to parse storage info'];

        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Get full server stats (CPU, RAM, Disk, Network)
     */
    public function getServerStats(int $serverId): array
    {
        $server = $this->getServer($serverId);
        if (!$server) {
            return ['success' => false, 'error' => 'Server not found'];
        }

        try {
            $password = $this->decryptPassword($server['password']);
            $path = $server['remote_path'] ?: '/';

            if ($server['type'] === 'ftp') {
                return ['success' => false, 'error' => 'Stats not supported for FTP'];
            }

            $port = $server['port'] ?: 22;

            // Combined command to get all stats
            $statsCommand = "echo '=====DISK=====' && df -B1 " . escapeshellarg($path) . " 2>/dev/null | tail -1 && " .
                           "echo '=====CPU=====' && top -bn1 | grep 'Cpu(s)' | head -1 && " .
                           "echo '=====MEM=====' && free -b | grep Mem && " .
                           "echo '=====NET=====' && cat /proc/net/dev | grep -E 'eth0|ens|enp' | head -1";

            $cmd = sprintf(
                'sshpass -p %s ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 -p %d %s@%s %s 2>&1',
                escapeshellarg($password),
                $port,
                escapeshellarg($server['username']),
                escapeshellarg($server['host']),
                escapeshellarg($statsCommand)
            );

            $output = shell_exec($cmd);

            if (empty($output)) {
                return ['success' => false, 'error' => 'Failed to get stats'];
            }

            $result = ['success' => true];

            // Parse Disk
            if (preg_match('/=====DISK=====\s*\n(.+)/m', $output, $m)) {
                $parts = preg_split('/\s+/', trim($m[1]));
                if (count($parts) >= 5 && is_numeric($parts[1])) {
                    $total = (float) $parts[1];
                    $used = (float) $parts[2];
                    $free = (float) $parts[3];
                    $result['disk'] = [
                        'total' => $this->formatBytes($total),
                        'used' => $this->formatBytes($used),
                        'free' => $this->formatBytes($free),
                        'percent' => $total > 0 ? round(($used / $total) * 100, 1) : 0
                    ];
                }
            }

            // Parse CPU: Cpu(s):  1.5%us,  0.5%sy,  0.0%ni, 97.8%id,  0.2%wa
            if (preg_match('/(\d+\.?\d*)%?\s*id/', $output, $m)) {
                $idle = (float) $m[1];
                $result['cpu'] = [
                    'used' => round(100 - $idle, 1),
                    'idle' => round($idle, 1)
                ];
            }

            // Parse Memory: Mem:  total  used  free  shared  buff/cache  available
            if (preg_match('/Mem:\s+(\d+)\s+(\d+)\s+(\d+)/', $output, $m)) {
                $total = (float) $m[1];
                $used = (float) $m[2];
                $free = (float) $m[3];
                $result['ram'] = [
                    'total' => $this->formatBytes($total),
                    'used' => $this->formatBytes($used),
                    'free' => $this->formatBytes($free),
                    'percent' => $total > 0 ? round(($used / $total) * 100, 1) : 0
                ];
            }

            // Parse Network (bytes received/transmitted)
            if (preg_match('/\w+:\s*(\d+)\s+\d+\s+\d+\s+\d+\s+\d+\s+\d+\s+\d+\s+\d+\s+(\d+)/', $output, $m)) {
                $result['network'] = [
                    'rx' => $this->formatBytes((float)$m[1]),
                    'tx' => $this->formatBytes((float)$m[2])
                ];
            }

            return $result;

        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Format bytes to human readable
     */
    private function formatBytes(float $bytes, int $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB', 'PB'];
        $bytes = max($bytes, 0);
        if ($bytes == 0) return '0 B';
        $pow = floor(log($bytes) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        return round($bytes, $precision) . ' ' . $units[$pow];
    }

    /**
     * Test SSH/SFTP connection
     */
    private function testSSHConnection(array $server): array
    {
        if (!function_exists('ssh2_connect')) {
            // Fallback to shell command
            return $this->testSSHViaShell($server);
        }

        $password = $this->decryptPassword($server['password']);

        $connection = @ssh2_connect($server['host'], $server['port'], [], [
            'disconnect' => function($reason, $message, $language) {
                // Handle disconnect
            }
        ]);

        if (!$connection) {
            return ['success' => false, 'error' => 'Could not connect to server'];
        }

        if (!empty($server['private_key'])) {
            $keyContent = $this->decryptPassword($server['private_key']);
            // Save temp key file
            $keyFile = tempnam(sys_get_temp_dir(), 'ssh_key_');
            file_put_contents($keyFile, $keyContent);
            chmod($keyFile, 0600);

            $auth = @ssh2_auth_pubkey_file($connection, $server['username'], $keyFile . '.pub', $keyFile, $password ?? '');
            unlink($keyFile);
        } else {
            $auth = @ssh2_auth_password($connection, $server['username'], $password);
        }

        if (!$auth) {
            return ['success' => false, 'error' => 'Authentication failed'];
        }

        // Test remote path exists
        $stream = @ssh2_exec($connection, "ls -la " . escapeshellarg($server['remote_path']));
        if ($stream) {
            stream_set_blocking($stream, true);
            $output = stream_get_contents($stream);
            fclose($stream);

            return [
                'success' => true,
                'message' => 'Connected successfully',
                'remote_path_exists' => strpos($output, 'total') !== false
            ];
        }

        return ['success' => true, 'message' => 'Connected successfully'];
    }

    /**
     * Test SSH via shell command (fallback)
     */
    private function testSSHViaShell(array $server): array
    {
        $password = $this->decryptPassword($server['password']);
        $privateKey = $this->decryptPassword($server['private_key']);

        // Create temp directory for SSH files if needed
        $tempDir = sys_get_temp_dir() . '/vod_ssh_' . posix_getuid();
        if (!is_dir($tempDir)) {
            @mkdir($tempDir, 0700, true);
        }

        // Check if private key is valid (should start with -----BEGIN)
        $hasValidKey = !empty($privateKey) && strpos($privateKey, '-----BEGIN') !== false;

        // Use sshpass for password auth or direct ssh for key auth
        if ($hasValidKey) {
            $keyFile = $tempDir . '/ssh_key_' . uniqid();
            file_put_contents($keyFile, $privateKey);
            chmod($keyFile, 0600);

            $cmd = sprintf(
                "ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -o ConnectTimeout=10 -o BatchMode=yes -i %s -p %d %s@%s 'echo connected' 2>&1",
                escapeshellarg($keyFile),
                $server['port'],
                escapeshellarg($server['username']),
                escapeshellarg($server['host'])
            );

            exec($cmd, $output, $returnCode);
            @unlink($keyFile);

            // If key auth failed but we have password, try password auth
            if ($returnCode !== 0 && !empty($password)) {
                $output = [];
                return $this->testSSHWithPassword($server, $password, $tempDir);
            }
        } else {
            // Use password authentication
            return $this->testSSHWithPassword($server, $password, $tempDir);
        }

        if ($returnCode === 0 && in_array('connected', $output)) {
            return ['success' => true, 'message' => 'Connected successfully via SSH'];
        }

        return ['success' => false, 'error' => implode("\n", $output)];
    }

    /**
     * Test SSH with password
     */
    private function testSSHWithPassword(array $server, ?string $password, string $tempDir): array
    {
        if (empty($password)) {
            return ['success' => false, 'error' => 'No valid SSH key or password provided'];
        }

        // Check if sshpass is available
        exec('which sshpass', $whichOutput, $whichCode);
        if ($whichCode !== 0) {
            return ['success' => false, 'error' => 'sshpass not installed. Install with: apt-get install sshpass'];
        }

        $cmd = sprintf(
            "sshpass -p %s ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -o ConnectTimeout=10 -o PreferredAuthentications=password -o PubkeyAuthentication=no -p %d %s@%s 'echo connected' 2>&1",
            escapeshellarg($password),
            $server['port'],
            escapeshellarg($server['username']),
            escapeshellarg($server['host'])
        );

        exec($cmd, $output, $returnCode);

        if ($returnCode === 0 && in_array('connected', $output)) {
            return ['success' => true, 'message' => 'Connected successfully via SSH (password)'];
        }

        return ['success' => false, 'error' => implode("\n", $output)];
    }

    /**
     * Test FTP connection
     */
    private function testFTPConnection(array $server): array
    {
        $password = $this->decryptPassword($server['password']);

        $conn = @ftp_connect($server['host'], $server['port'], 10);
        if (!$conn) {
            return ['success' => false, 'error' => 'Could not connect to FTP server'];
        }

        $login = @ftp_login($conn, $server['username'], $password);
        if (!$login) {
            ftp_close($conn);
            return ['success' => false, 'error' => 'FTP authentication failed'];
        }

        // Enable passive mode
        ftp_pasv($conn, true);

        // Test remote path
        $pathExists = @ftp_chdir($conn, $server['remote_path']);

        ftp_close($conn);

        return [
            'success' => true,
            'message' => 'FTP connected successfully',
            'remote_path_exists' => $pathExists
        ];
    }

    /**
     * Download file to remote server
     */
    public function downloadToRemote(int $remoteServerId, string $sourceUrl, string $filename, ?int $movieId = null, ?int $episodeId = null): array
    {
        $server = $this->getServer($remoteServerId);
        if (!$server) {
            return ['success' => false, 'error' => 'Remote server not found'];
        }

        // If filename starts with /, treat as absolute path, otherwise prepend remote_path
        if (strpos($filename, '/') === 0) {
            $destinationPath = $filename;
        } else {
            $destinationPath = rtrim($server['remote_path'], '/') . '/' . $filename;
        }

        // Create download job
        $stmt = $this->db->prepare("
            INSERT INTO download_jobs (movie_id, episode_id, remote_server_id, source_url, filename, status, created_at)
            VALUES (?, ?, ?, ?, ?, 'pending', NOW())
        ");
        $stmt->execute([$movieId, $episodeId, $remoteServerId, $sourceUrl, $destinationPath]);
        $jobId = (int) $this->db->lastInsertId();

        // Start download
        try {
            switch ($server['type']) {
                case 'ssh':
                case 'sftp':
                    $result = $this->downloadViaSSH($server, $sourceUrl, $destinationPath, $jobId);
                    break;
                case 'ftp':
                    $result = $this->downloadViaFTP($server, $sourceUrl, $destinationPath, $jobId);
                    break;
                default:
                    $result = ['success' => false, 'error' => 'Unknown server type'];
            }

            // Update job status
            $status = $result['success'] ? 'completed' : 'failed';
            $this->updateJobStatus($jobId, $status, $result['error'] ?? null);

            return array_merge(['job_id' => $jobId], $result);

        } catch (Exception $e) {
            $this->updateJobStatus($jobId, 'failed', $e->getMessage());
            return ['success' => false, 'job_id' => $jobId, 'error' => $e->getMessage()];
        }
    }

    /**
     * Download via SSH (using wget/curl on remote server)
     */
    private function downloadViaSSH(array $server, string $sourceUrl, string $destinationPath, int $jobId): array
    {
        $password = $this->decryptPassword($server['password']);
        $privateKey = $this->decryptPassword($server['private_key']);

        // Create temp directory for SSH files if needed
        $tempDir = sys_get_temp_dir() . '/vod_ssh_' . posix_getuid();
        if (!is_dir($tempDir)) {
            @mkdir($tempDir, 0700, true);
        }

        // Create directory on remote server
        $dir = dirname($destinationPath);

        // Build SSH command with Smart Retry & Protocol Switching
        // User-Agent: TiviMate/5.1.6 (Android 7.1.2)
        // Max retries: 10, Delay: 5s
        // Switch to wget on: 403, 429, Connection Reset, Timeout
        $userAgent = 'TiviMate/5.1.6 (Android 7.1.2)';
        $maxRetries = 10;
        $retryDelay = 5;

        $downloadCmd = sprintf(
            'mkdir -p %s && ' .
            '(' .
            // Try wget first with retries
            'for i in $(seq 1 %d); do ' .
            'wget --tries=3 --timeout=60 --waitretry=%d --retry-connrefused ' .
            '--user-agent=%s -q -O %s %s && exit 0; ' .
            'sleep %d; done; ' .
            // Fallback to curl with retries
            'for i in $(seq 1 %d); do ' .
            'curl -L -k --retry 3 --retry-delay %d --max-time 300 ' .
            '-A %s -o %s %s && exit 0; ' .
            'sleep %d; done; ' .
            'exit 1' .
            ') 2>&1',
            escapeshellarg($dir),
            $maxRetries,
            $retryDelay,
            escapeshellarg($userAgent),
            escapeshellarg($destinationPath),
            escapeshellarg($sourceUrl),
            $retryDelay,
            $maxRetries,
            $retryDelay,
            escapeshellarg($userAgent),
            escapeshellarg($destinationPath),
            escapeshellarg($sourceUrl),
            $retryDelay
        );

        // Check if private key is valid
        $hasValidKey = !empty($privateKey) && strpos($privateKey, '-----BEGIN') !== false;

        // Update status to downloading
        $this->updateJobStatus($jobId, 'downloading');

        if ($hasValidKey) {
            $keyFile = $tempDir . '/ssh_key_' . uniqid();
            file_put_contents($keyFile, $privateKey);
            chmod($keyFile, 0600);

            $cmd = sprintf(
                "ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -o ConnectTimeout=30 -o BatchMode=yes -i %s -p %d %s@%s %s 2>&1",
                escapeshellarg($keyFile),
                $server['port'],
                escapeshellarg($server['username']),
                escapeshellarg($server['host']),
                escapeshellarg($downloadCmd)
            );

            exec($cmd, $output, $returnCode);
            @unlink($keyFile);

            // If key auth failed, try password
            if ($returnCode !== 0 && !empty($password)) {
                $output = [];
                return $this->downloadViaSSHPassword($server, $password, $downloadCmd, $jobId);
            }
        } else {
            // Use password authentication
            return $this->downloadViaSSHPassword($server, $password, $downloadCmd, $jobId);
        }

        $outputStr = implode("\n", $output);

        if ($returnCode === 0) {
            $this->logger->info("Download completed via SSH", [
                'job_id' => $jobId,
                'destination' => $destinationPath
            ]);
            return ['success' => true, 'message' => 'Download completed', 'output' => $outputStr];
        }

        return ['success' => false, 'error' => $outputStr];
    }

    /**
     * Download via SSH with password
     */
    private function downloadViaSSHPassword(array $server, ?string $password, string $downloadCmd, int $jobId): array
    {
        if (empty($password)) {
            return ['success' => false, 'error' => 'No valid SSH key or password provided'];
        }

        exec('which sshpass', $whichOutput, $whichCode);
        if ($whichCode !== 0) {
            return ['success' => false, 'error' => 'sshpass not installed'];
        }

        $cmd = sprintf(
            "sshpass -p %s ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -o ConnectTimeout=30 -o PreferredAuthentications=password -o PubkeyAuthentication=no -p %d %s@%s %s 2>&1",
            escapeshellarg($password),
            $server['port'],
            escapeshellarg($server['username']),
            escapeshellarg($server['host']),
            escapeshellarg($downloadCmd)
        );

        exec($cmd, $output, $returnCode);
        $outputStr = implode("\n", $output);

        if ($returnCode === 0) {
            $this->logger->info("Download completed via SSH (password)", ['job_id' => $jobId]);
            return ['success' => true, 'message' => 'Download completed', 'output' => $outputStr];
        }

        return ['success' => false, 'error' => $outputStr];
    }

    /**
     * Download via FTP
     */
    private function downloadViaFTP(array $server, string $sourceUrl, string $destinationPath, int $jobId): array
    {
        // First download locally using Smart Retry, then upload via FTP
        $tempFile = tempnam(sys_get_temp_dir(), 'ftp_download_');
        $userAgent = 'TiviMate/5.1.6 (Android 7.1.2)';
        $maxRetries = 10;
        $retryDelay = 5;

        $this->updateJobStatus($jobId, 'downloading');

        // Smart download with retry and protocol switching
        $success = false;
        $error = '';

        // Try curl first with retries
        for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
            $ch = curl_init($sourceUrl);
            $fp = fopen($tempFile, 'w');

            curl_setopt_array($ch, [
                CURLOPT_FILE => $fp,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_TIMEOUT => 3600,
                CURLOPT_CONNECTTIMEOUT => 60,
                CURLOPT_USERAGENT => $userAgent,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => false,
                CURLOPT_HTTPHEADER => [
                    'Accept: */*',
                    'Connection: keep-alive'
                ],
                CURLOPT_NOPROGRESS => false,
                CURLOPT_PROGRESSFUNCTION => function($ch, $downloadSize, $downloaded, $uploadSize, $uploaded) use ($jobId) {
                    if ($downloadSize > 0) {
                        $progress = round(($downloaded / $downloadSize) * 100);
                        $this->updateJobProgress($jobId, $progress, $downloaded, $downloadSize);
                    }
                }
            ]);

            $success = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            fclose($fp);

            // Check if success
            if ($success && $httpCode == 200 && filesize($tempFile) > 0) {
                $success = true;
                break;
            }

            // Check for retryable errors (403, 429, timeout, connection reset)
            $retryable = in_array($httpCode, [403, 429, 500, 502, 503, 504]) ||
                         strpos($error, 'timed out') !== false ||
                         strpos($error, 'reset') !== false ||
                         strpos($error, 'refused') !== false;

            if (!$retryable && $attempt >= 3) {
                // Switch to wget via shell
                $this->logger->info("Switching to wget (attempt $attempt)", ['error' => $error]);
                break;
            }

            $this->logger->warning("Download attempt $attempt failed: $error (HTTP $httpCode)");
            if ($attempt < $maxRetries) {
                sleep($retryDelay);
            }
        }

        // If curl failed, try wget
        if (!$success || !file_exists($tempFile) || filesize($tempFile) == 0) {
            @unlink($tempFile);

            for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
                $cmd = sprintf(
                    'wget --tries=3 --timeout=60 --waitretry=%d --retry-connrefused --user-agent=%s -q -O %s %s 2>&1',
                    $retryDelay,
                    escapeshellarg($userAgent),
                    escapeshellarg($tempFile),
                    escapeshellarg($sourceUrl)
                );

                exec($cmd, $output, $returnCode);

                if ($returnCode === 0 && file_exists($tempFile) && filesize($tempFile) > 0) {
                    $success = true;
                    break;
                }

                $error = implode("\n", $output);
                $this->logger->warning("wget attempt $attempt failed: $error");
                $output = [];

                if ($attempt < $maxRetries) {
                    sleep($retryDelay);
                }
            }
        }

        if (!$success || !file_exists($tempFile) || filesize($tempFile) == 0) {
            @unlink($tempFile);
            return ['success' => false, 'error' => 'Download failed after all retries: ' . $error];
        }

        // Upload to FTP
        $password = $this->decryptPassword($server['password']);

        $conn = @ftp_connect($server['host'], $server['port'], 30);
        if (!$conn) {
            unlink($tempFile);
            return ['success' => false, 'error' => 'Could not connect to FTP server'];
        }

        if (!@ftp_login($conn, $server['username'], $password)) {
            ftp_close($conn);
            unlink($tempFile);
            return ['success' => false, 'error' => 'FTP authentication failed'];
        }

        ftp_pasv($conn, true);

        // Create directory structure
        $dir = dirname($destinationPath);
        $this->ftpMkdirRecursive($conn, $dir);

        // Upload file
        $uploaded = @ftp_put($conn, $destinationPath, $tempFile, FTP_BINARY);

        ftp_close($conn);
        unlink($tempFile);

        if ($uploaded) {
            $this->logger->info("Download completed via FTP", [
                'job_id' => $jobId,
                'destination' => $destinationPath
            ]);
            return ['success' => true, 'message' => 'Download completed'];
        }

        return ['success' => false, 'error' => 'FTP upload failed'];
    }

    /**
     * Create FTP directory recursively
     */
    private function ftpMkdirRecursive($conn, string $dir): void
    {
        $parts = explode('/', trim($dir, '/'));
        $path = '';

        foreach ($parts as $part) {
            $path .= '/' . $part;
            @ftp_mkdir($conn, $path);
        }
    }

    /**
     * Update job status
     */
    private function updateJobStatus(int $jobId, string $status, ?string $error = null): void
    {
        $sql = "UPDATE download_jobs SET status = ?, error_message = ?";

        if ($status === 'downloading') {
            $sql .= ", started_at = NOW()";
        } elseif (in_array($status, ['completed', 'failed'])) {
            $sql .= ", completed_at = NOW()";
        }

        $sql .= " WHERE id = ?";

        $stmt = $this->db->prepare($sql);
        $stmt->execute([$status, $error, $jobId]);
    }

    /**
     * Update job progress
     */
    private function updateJobProgress(int $jobId, int $progress, int $downloaded, int $total): void
    {
        $stmt = $this->db->prepare("
            UPDATE download_jobs SET progress = ?, downloaded_bytes = ?, total_bytes = ?
            WHERE id = ?
        ");
        $stmt->execute([$progress, $downloaded, $total, $jobId]);
    }

    /**
     * Get download jobs
     */
    public function getJobs(?int $remoteServerId = null, ?string $status = null, int $limit = 50): array
    {
        $sql = "SELECT j.*, r.name as server_name, r.host FROM download_jobs j
                JOIN remote_servers r ON j.remote_server_id = r.id";

        $where = [];
        $params = [];

        if ($remoteServerId) {
            $where[] = "j.remote_server_id = ?";
            $params[] = $remoteServerId;
        }

        if ($status) {
            $where[] = "j.status = ?";
            $params[] = $status;
        }

        if ($where) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }

        $sql .= " ORDER BY j.created_at DESC LIMIT ?";
        $params[] = $limit;

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll();
    }

    /**
     * Get pending jobs for a server
     */
    public function getPendingJobs(int $remoteServerId): array
    {
        $stmt = $this->db->prepare("
            SELECT * FROM download_jobs
            WHERE remote_server_id = ? AND status IN ('pending', 'downloading')
            ORDER BY created_at ASC
        ");
        $stmt->execute([$remoteServerId]);

        return $stmt->fetchAll();
    }

    /**
     * Encrypt password/sensitive data
     */
    private function encryptPassword(string $data): string
    {
        $key = hash('sha256', DB_PASS . 'vod_encrypt_key', true);
        $iv = random_bytes(16);
        $encrypted = openssl_encrypt($data, 'AES-256-CBC', $key, 0, $iv);
        return base64_encode($iv . $encrypted);
    }

    /**
     * Decrypt password/sensitive data
     */
    private function decryptPassword(?string $data): ?string
    {
        if (empty($data)) {
            return null;
        }

        $key = hash('sha256', DB_PASS . 'vod_encrypt_key', true);
        $decoded = base64_decode($data);
        $iv = substr($decoded, 0, 16);
        $encrypted = substr($decoded, 16);

        return openssl_decrypt($encrypted, 'AES-256-CBC', $key, 0, $iv);
    }

    /**
     * List folders on remote server
     */
    public function listRemoteFolders(int $remoteServerId, ?string $path = '/'): array
    {
        $server = $this->getServer($remoteServerId);
        if (!$server) {
            return ['error' => 'Server not found'];
        }

        // Always use provided path, default to root
        $basePath = $path ?: '/';
        $basePath = rtrim($basePath, '/') ?: '/';

        if ($server['type'] === 'ftp') {
            return $this->listFTPFolders($server, $basePath);
        }

        return $this->listSSHFolders($server, $basePath);
    }

    /**
     * List folders via SSH
     */
    private function listSSHFolders(array $server, string $path): array
    {
        $password = $this->decryptPassword($server['password']);
        $privateKey = $this->decryptPassword($server['private_key']);

        $tempDir = sys_get_temp_dir() . '/vod_ssh_' . posix_getuid();
        if (!is_dir($tempDir)) {
            @mkdir($tempDir, 0700, true);
        }

        $hasValidKey = !empty($privateKey) && strpos($privateKey, '-----BEGIN') !== false;

        $listCmd = "ls -la " . escapeshellarg($path) . " 2>/dev/null | grep '^d' | awk '{print \$NF}' | grep -v '^\\.\\.\?$'";

        if ($hasValidKey) {
            $keyFile = $tempDir . '/ssh_key_' . uniqid();
            file_put_contents($keyFile, $privateKey);
            chmod($keyFile, 0600);

            $cmd = sprintf(
                "ssh -o StrictHostKeyChecking=accept-new -o UserKnownHostsFile=/dev/null -o ConnectTimeout=10 -o BatchMode=yes -o LogLevel=ERROR -i %s -p %d %s@%s %s 2>/dev/null",
                escapeshellarg($keyFile),
                $server['port'],
                escapeshellarg($server['username']),
                escapeshellarg($server['host']),
                escapeshellarg($listCmd)
            );

            exec($cmd, $output, $returnCode);
            @unlink($keyFile);

            if ($returnCode !== 0 && !empty($password)) {
                $output = [];
                return $this->listSSHFoldersPassword($server, $password, $path);
            }
        } else {
            return $this->listSSHFoldersPassword($server, $password, $path);
        }

        // Filter out SSH warnings and empty lines (PHP 7.4 compatible)
        $folderNames = array_filter($output, function($f) {
            $f = trim($f);
            return !empty($f) && strpos($f, 'Warning:') === false && strpos($f, 'known_hosts') === false;
        });

        // Format folders with name and full path
        $folders = [];
        foreach ($folderNames as $name) {
            $name = trim($name);
            if ($name && $name !== '.' && $name !== '..') {
                $folders[] = [
                    'name' => $name,
                    'path' => $path . '/' . $name
                ];
            }
        }

        // Calculate parent path
        $parentPath = dirname($path);
        if ($parentPath === $path || $path === '/') {
            $parentPath = null;
        }

        return [
            'success' => true,
            'current_path' => $path,
            'parent_path' => $parentPath,
            'folders' => $folders
        ];
    }

    /**
     * List folders via SSH with password
     */
    private function listSSHFoldersPassword(array $server, ?string $password, string $path): array
    {
        if (empty($password)) {
            return ['error' => 'No credentials available'];
        }

        exec('which sshpass', $whichOutput, $whichCode);
        if ($whichCode !== 0) {
            return ['error' => 'sshpass not installed'];
        }

        $listCmd = "ls -la " . escapeshellarg($path) . " 2>/dev/null | grep '^d' | awk '{print \$NF}' | grep -v '^\\.\\.\?$'";

        $cmd = sprintf(
            "sshpass -p %s ssh -o StrictHostKeyChecking=accept-new -o UserKnownHostsFile=/dev/null -o ConnectTimeout=10 -o PreferredAuthentications=password -o PubkeyAuthentication=no -o LogLevel=ERROR -p %d %s@%s %s 2>/dev/null",
            escapeshellarg($password),
            $server['port'],
            escapeshellarg($server['username']),
            escapeshellarg($server['host']),
            escapeshellarg($listCmd)
        );

        exec($cmd, $output, $returnCode);

        // Filter out SSH warnings and empty lines (PHP 7.4 compatible)
        $folderNames = array_filter($output, function($f) {
            $f = trim($f);
            return !empty($f) && strpos($f, 'Permission denied') === false && strpos($f, 'Warning:') === false && strpos($f, 'known_hosts') === false;
        });

        // Format folders with name and full path
        $folders = [];
        foreach ($folderNames as $name) {
            $name = trim($name);
            if ($name && $name !== '.' && $name !== '..') {
                $folders[] = [
                    'name' => $name,
                    'path' => $path . '/' . $name
                ];
            }
        }

        // Calculate parent path
        $parentPath = dirname($path);
        if ($parentPath === $path || $path === '/') {
            $parentPath = null;
        }

        return [
            'success' => true,
            'current_path' => $path,
            'parent_path' => $parentPath,
            'folders' => $folders
        ];
    }

    /**
     * List folders via FTP
     */
    private function listFTPFolders(array $server, string $path): array
    {
        $password = $this->decryptPassword($server['password']);

        $conn = ftp_connect($server['host'], $server['port'] ?: 21, 10);
        if (!$conn) {
            return ['error' => 'FTP connection failed'];
        }

        if (!ftp_login($conn, $server['username'], $password)) {
            ftp_close($conn);
            return ['error' => 'FTP login failed'];
        }

        ftp_pasv($conn, true);

        $items = ftp_nlist($conn, $path);
        ftp_close($conn);

        if ($items === false) {
            return ['error' => 'Failed to list directory'];
        }

        $folders = [];
        foreach ($items as $item) {
            $name = basename($item);
            if ($name !== '.' && $name !== '..') {
                $folders[] = [
                    'name' => $name,
                    'path' => $path . '/' . $name
                ];
            }
        }

        // Calculate parent path
        $parentPath = dirname($path);
        if ($parentPath === $path || $path === '/') {
            $parentPath = null;
        }

        return [
            'success' => true,
            'current_path' => $path,
            'parent_path' => $parentPath,
            'folders' => $folders
        ];
    }

    /**
     * Create folder on remote server
     */
    public function createRemoteFolder(int $remoteServerId, string $path): array
    {
        $server = $this->getServer($remoteServerId);
        if (!$server) {
            return ['error' => 'Server not found'];
        }

        $password = $this->decryptPassword($server['password']);
        $privateKey = $this->decryptPassword($server['private_key']);

        if ($server['type'] === 'ftp') {
            $conn = ftp_connect($server['host'], $server['port'] ?: 21, 10);
            if (!$conn || !ftp_login($conn, $server['username'], $password)) {
                return ['error' => 'FTP connection failed'];
            }
            ftp_pasv($conn, true);
            $result = @ftp_mkdir($conn, $path);
            ftp_close($conn);
            return $result ? ['success' => true, 'path' => $path] : ['error' => 'Failed to create folder'];
        }

        // SSH
        $tempDir = sys_get_temp_dir() . '/vod_ssh_' . posix_getuid();
        if (!is_dir($tempDir)) {
            @mkdir($tempDir, 0700, true);
        }

        $hasValidKey = !empty($privateKey) && strpos($privateKey, '-----BEGIN') !== false;
        $mkdirCmd = "mkdir -p " . escapeshellarg($path);

        if ($hasValidKey) {
            $keyFile = $tempDir . '/ssh_key_' . uniqid();
            file_put_contents($keyFile, $privateKey);
            chmod($keyFile, 0600);

            $cmd = sprintf(
                "ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -o ConnectTimeout=10 -o BatchMode=yes -i %s -p %d %s@%s %s 2>&1",
                escapeshellarg($keyFile),
                $server['port'],
                escapeshellarg($server['username']),
                escapeshellarg($server['host']),
                escapeshellarg($mkdirCmd)
            );

            exec($cmd, $output, $returnCode);
            @unlink($keyFile);

            if ($returnCode !== 0 && !empty($password)) {
                exec('which sshpass', $whichOutput, $whichCode);
                if ($whichCode === 0) {
                    $cmd = sprintf(
                        "sshpass -p %s ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -o ConnectTimeout=10 -o PreferredAuthentications=password -o PubkeyAuthentication=no -p %d %s@%s %s 2>&1",
                        escapeshellarg($password),
                        $server['port'],
                        escapeshellarg($server['username']),
                        escapeshellarg($server['host']),
                        escapeshellarg($mkdirCmd)
                    );
                    exec($cmd, $output, $returnCode);
                }
            }
        } else if (!empty($password)) {
            exec('which sshpass', $whichOutput, $whichCode);
            if ($whichCode !== 0) {
                return ['error' => 'sshpass not installed'];
            }

            $cmd = sprintf(
                "sshpass -p %s ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -o ConnectTimeout=10 -o PreferredAuthentications=password -o PubkeyAuthentication=no -p %d %s@%s %s 2>&1",
                escapeshellarg($password),
                $server['port'],
                escapeshellarg($server['username']),
                escapeshellarg($server['host']),
                escapeshellarg($mkdirCmd)
            );

            exec($cmd, $output, $returnCode);
        } else {
            return ['error' => 'No credentials available'];
        }

        return $returnCode === 0
            ? ['success' => true, 'path' => $path]
            : ['error' => 'Failed to create folder: ' . implode("\n", $output)];
    }

    /**
     * Bulk download to remote server
     */
    public function bulkDownload(int $remoteServerId, array $items): array
    {
        $results = [
            'queued' => 0,
            'errors' => 0,
            'jobs' => []
        ];

        foreach ($items as $item) {
            // Build filename - just use subfolder + filename directly
            $filename = $item['filename'];

            if (!empty($item['subfolder'])) {
                $subfolder = $item['subfolder'];
                // Check if subfolder is an absolute path
                if (strpos($subfolder, '/') === 0) {
                    // Absolute path - use directly: /path/to/folder/filename
                    $filename = rtrim($subfolder, '/') . '/' . $filename;
                } else {
                    // Relative path
                    $subfolder = trim($subfolder, '/');
                    $filename = $subfolder . '/' . $filename;
                }
            }
            // No more organize_by_name - files go directly to selected folder

            $result = $this->downloadToRemote(
                $remoteServerId,
                $item['url'],
                $filename,
                $item['movie_id'] ?? null,
                $item['episode_id'] ?? null
            );

            if (isset($result['job_id'])) {
                $results['queued']++;
                $results['jobs'][] = $result['job_id'];
            } else {
                $results['errors']++;
            }
        }

        return $results;
    }
}
