<?php
/**
 * Movie Importer
 * Handles importing movies from multiple sources with smart sync
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/ServerManager.php';
require_once __DIR__ . '/XtreamAPI.php';
require_once __DIR__ . '/M3UParser.php';
require_once __DIR__ . '/Logger.php';

class MovieImporter
{
    private PDO $db;
    private Logger $logger;
    private ServerManager $serverManager;

    public function __construct()
    {
        $this->db = getDB();
        $this->logger = new Logger('importer', $this->db);
        $this->serverManager = new ServerManager();
    }

    /**
     * Import movies from a specific server
     */
    public function importFromServer(int $serverId, ?int $categoryId = null, bool $forceUpdate = false): array
    {
        $server = $this->serverManager->getServer($serverId);

        if (!$server) {
            return ['success' => false, 'error' => 'Server not found'];
        }

        $this->logger->info("Starting import from server", [
            'server_id' => $serverId,
            'server_name' => $server['name'],
            'category_id' => $categoryId
        ]);

        if ($server['type'] === 'xtream') {
            return $this->importFromXtream($server, $categoryId, $forceUpdate);
        } else {
            return $this->importFromM3U($server, $categoryId, $forceUpdate);
        }
    }

    /**
     * Import from Xtream Codes server
     */
    private function importFromXtream(array $server, ?int $categoryId, bool $forceUpdate): array
    {
        $result = [
            'success' => true,
            'imported' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => 0,
            'movies' => []
        ];

        $api = new XtreamAPI(
            $server['url'],
            $server['username'],
            $server['password'],
            $server['port'],
            (bool) $server['use_https']
        );

        if (!$api->authenticate()) {
            $result['success'] = false;
            $result['error'] = 'Authentication failed';
            return $result;
        }

        // Get VOD streams
        $movies = $api->getVODStreams($categoryId);
        unset($movies['_meta']);

        $this->logger->info("Fetched movies from Xtream", ['count' => count($movies)]);

        foreach ($movies as $movie) {
            if (!is_array($movie) || !isset($movie['stream_id'])) {
                continue;
            }

            try {
                $importResult = $this->processMovie($movie, $server, $api, $forceUpdate);

                switch ($importResult['action']) {
                    case 'imported':
                        $result['imported']++;
                        break;
                    case 'updated':
                        $result['updated']++;
                        break;
                    case 'skipped':
                        $result['skipped']++;
                        break;
                }

                if (isset($importResult['movie_id'])) {
                    $result['movies'][] = $importResult['movie_id'];
                }

            } catch (Exception $e) {
                $result['errors']++;
                $this->logger->error("Failed to import movie", [
                    'movie_name' => $movie['name'] ?? 'Unknown',
                    'error' => $e->getMessage()
                ]);
            }
        }

        // Update server last sync
        $this->serverManager->updateServer($server['id'], [
            'last_sync' => date('Y-m-d H:i:s')
        ]);

        $this->logger->info("Import completed", $result);

        return $result;
    }

    /**
     * Import from M3U playlist
     */
    private function importFromM3U(array $server, ?string $categoryFilter, bool $forceUpdate): array
    {
        $result = [
            'success' => true,
            'imported' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => 0,
            'movies' => []
        ];

        $parser = new M3UParser();
        $parseResult = $parser->parseUrl($server['url']);

        if (!$parseResult['success']) {
            $result['success'] = false;
            $result['error'] = $parseResult['error'];
            return $result;
        }

        $entries = $parseResult['entries'];

        // Filter by category if specified
        if ($categoryFilter) {
            $entries = array_filter($entries, function ($entry) use ($categoryFilter) {
                return ($entry['group_title'] ?? '') === $categoryFilter;
            });
        }

        // Filter only movies
        $movies = array_filter($entries, function ($entry) {
            return ($entry['stream_type'] ?? '') === 'movie';
        });

        $this->logger->info("Parsed M3U playlist", ['total' => count($movies)]);

        foreach ($movies as $movie) {
            try {
                $importResult = $this->processM3UMovie($movie, $server, $forceUpdate);

                switch ($importResult['action']) {
                    case 'imported':
                        $result['imported']++;
                        break;
                    case 'updated':
                        $result['updated']++;
                        break;
                    case 'skipped':
                        $result['skipped']++;
                        break;
                }

                if (isset($importResult['movie_id'])) {
                    $result['movies'][] = $importResult['movie_id'];
                }

            } catch (Exception $e) {
                $result['errors']++;
                $this->logger->error("Failed to import M3U movie", [
                    'movie_name' => $movie['name'] ?? 'Unknown',
                    'error' => $e->getMessage()
                ]);
            }
        }

        // Update server last sync
        $this->serverManager->updateServer($server['id'], [
            'last_sync' => date('Y-m-d H:i:s')
        ]);

        return $result;
    }

    /**
     * Process a single movie from Xtream
     */
    private function processMovie(array $vodData, array $server, XtreamAPI $api, bool $forceUpdate): array
    {
        $streamId = $vodData['stream_id'];

        // Transform to standard format
        $movieData = $api->transformVODToStandard($vodData);

        // Check if movie already exists (by title+year or by link)
        $existingMovie = $this->findExistingMovie($movieData['name'], $movieData['year']);

        // Check if link already exists
        $existingLink = $this->findExistingLink($server['id'], $streamId);

        if ($existingLink && !$forceUpdate) {
            return ['action' => 'skipped', 'movie_id' => $existingLink['movie_id']];
        }

        if ($existingMovie) {
            // Update existing movie and add new link
            $movieId = $existingMovie['id'];

            if ($forceUpdate) {
                $this->updateMovie($movieId, $movieData);
            }

            // Add link if it doesn't exist
            if (!$existingLink) {
                $this->addMovieLink($movieId, $server, $vodData, $api);
            }

            return ['action' => 'updated', 'movie_id' => $movieId];

        } else {
            // Create new movie
            $movieId = $this->createMovie($movieData);

            // Add link
            $this->addMovieLink($movieId, $server, $vodData, $api);

            return ['action' => 'imported', 'movie_id' => $movieId];
        }
    }

    /**
     * Process a single movie from M3U
     */
    private function processM3UMovie(array $entry, array $server, bool $forceUpdate): array
    {
        $name = $entry['parsed_title'] ?? $entry['name'];
        $year = $entry['parsed_year'];
        $streamId = $entry['stream_id'] ?? crc32($entry['url']);

        // Check if movie exists
        $existingMovie = $this->findExistingMovie($name, $year);

        // Check if link exists
        $existingLink = $this->findExistingLink($server['id'], $streamId);

        if ($existingLink && !$forceUpdate) {
            return ['action' => 'skipped', 'movie_id' => $existingLink['movie_id']];
        }

        $movieData = [
            'name' => $name,
            'year' => $year,
            'clean_name' => $entry['clean_name'],
            'poster' => $entry['tvg_logo'] ?? null,
            'container_extension' => $entry['container'] ?? 'mp4'
        ];

        if ($existingMovie) {
            $movieId = $existingMovie['id'];

            if (!$existingLink) {
                $this->addM3UMovieLink($movieId, $server, $entry);
            }

            return ['action' => 'updated', 'movie_id' => $movieId];

        } else {
            $movieId = $this->createMovie($movieData);
            $this->addM3UMovieLink($movieId, $server, $entry);

            return ['action' => 'imported', 'movie_id' => $movieId];
        }
    }

    /**
     * Find existing movie by title and year
     */
    private function findExistingMovie(string $title, ?int $year): ?array
    {
        $sql = "SELECT * FROM movies WHERE title = :title";
        $params = ['title' => $title];

        if ($year) {
            $sql .= " AND year = :year";
            $params['year'] = $year;
        }

        $sql .= " LIMIT 1";

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetch() ?: null;
    }

    /**
     * Find existing link
     */
    private function findExistingLink(int $serverId, int $streamId): ?array
    {
        $stmt = $this->db->prepare("
            SELECT * FROM movie_links
            WHERE server_id = :server_id AND stream_id = :stream_id
            LIMIT 1
        ");

        $stmt->execute([
            'server_id' => $serverId,
            'stream_id' => $streamId
        ]);

        return $stmt->fetch() ?: null;
    }

    /**
     * Create new movie
     */
    private function createMovie(array $data): int
    {
        $stmt = $this->db->prepare("
            INSERT INTO movies (
                title, original_title, year, clean_name, description,
                genre, rating, rating_5based, poster, container_extension,
                is_adult, added_timestamp
            ) VALUES (
                :title, :original_title, :year, :clean_name, :description,
                :genre, :rating, :rating_5based, :poster, :container_extension,
                :is_adult, :added_timestamp
            )
        ");

        $stmt->execute([
            'title' => $data['name'],
            'original_title' => $data['original_name'] ?? null,
            'year' => $data['year'],
            'clean_name' => $data['clean_name'] ?? $this->generateCleanName($data['name'], $data['year']),
            'description' => $data['description'] ?? null,
            'genre' => $data['genre'] ?? null,
            'rating' => $data['rating'] ?? null,
            'rating_5based' => $data['rating_5based'] ?? null,
            'poster' => $data['poster'] ?? null,
            'container_extension' => $data['container_extension'] ?? 'mp4',
            'is_adult' => $data['is_adult'] ?? 0,
            'added_timestamp' => $data['added_timestamp'] ?? time()
        ]);

        return (int) $this->db->lastInsertId();
    }

    /**
     * Update existing movie
     */
    private function updateMovie(int $movieId, array $data): void
    {
        $stmt = $this->db->prepare("
            UPDATE movies SET
                original_title = COALESCE(:original_title, original_title),
                description = COALESCE(:description, description),
                genre = COALESCE(:genre, genre),
                rating = COALESCE(:rating, rating),
                poster = COALESCE(:poster, poster)
            WHERE id = :id
        ");

        $stmt->execute([
            'id' => $movieId,
            'original_title' => $data['original_name'] ?? null,
            'description' => $data['description'] ?? null,
            'genre' => $data['genre'] ?? null,
            'rating' => $data['rating'] ?? null,
            'poster' => $data['poster'] ?? null
        ]);
    }

    /**
     * Add movie link from Xtream
     */
    private function addMovieLink(int $movieId, array $server, array $vodData, XtreamAPI $api): int
    {
        $extension = $vodData['container_extension'] ?? 'mp4';
        $streamUrl = $api->getVODStreamUrl($vodData['stream_id'], $extension);
        $categoryId = $vodData['category_id'] ?? null;

        $stmt = $this->db->prepare("
            INSERT INTO movie_links (
                movie_id, server_id, stream_id, stream_url,
                container_extension, added_timestamp, priority, category_id
            ) VALUES (
                :movie_id, :server_id, :stream_id, :stream_url,
                :container_extension, :added_timestamp, :priority, :category_id
            )
        ");

        $stmt->execute([
            'movie_id' => $movieId,
            'server_id' => $server['id'],
            'stream_id' => $vodData['stream_id'],
            'stream_url' => $streamUrl,
            'container_extension' => $extension,
            'added_timestamp' => $vodData['added'] ?? time(),
            'priority' => $server['priority'] ?? 1,
            'category_id' => $categoryId
        ]);

        return (int) $this->db->lastInsertId();
    }

    /**
     * Add movie link from M3U
     */
    private function addM3UMovieLink(int $movieId, array $server, array $entry): int
    {
        $streamId = $entry['stream_id'] ?? crc32($entry['url']);

        $stmt = $this->db->prepare("
            INSERT INTO movie_links (
                movie_id, server_id, stream_id, stream_url,
                container_extension, priority
            ) VALUES (
                :movie_id, :server_id, :stream_id, :stream_url,
                :container_extension, :priority
            )
            ON DUPLICATE KEY UPDATE stream_url = VALUES(stream_url)
        ");

        $stmt->execute([
            'movie_id' => $movieId,
            'server_id' => $server['id'],
            'stream_id' => $streamId,
            'stream_url' => $entry['url'],
            'container_extension' => $entry['container'] ?? 'mp4',
            'priority' => $server['priority'] ?? 1
        ]);

        return (int) $this->db->lastInsertId();
    }

    /**
     * Generate clean filename for FolderWatch
     */
    private function generateCleanName(string $title, ?int $year, string $extension = 'mp4'): string
    {
        $cleanTitle = preg_replace('/[^\w\s\-]/u', '', $title);
        $cleanTitle = preg_replace('/\s+/', '.', trim($cleanTitle));

        if ($year) {
            return "{$cleanTitle} ({$year}).{$extension}";
        }

        return "{$cleanTitle}.{$extension}";
    }

    /**
     * Import single movie by stream ID
     */
    public function importSingleMovie(int $serverId, int $streamId): array
    {
        $server = $this->serverManager->getServer($serverId);

        if (!$server || $server['type'] !== 'xtream') {
            return ['success' => false, 'error' => 'Invalid server'];
        }

        $api = new XtreamAPI(
            $server['url'],
            $server['username'],
            $server['password'],
            $server['port'],
            (bool) $server['use_https']
        );

        if (!$api->authenticate()) {
            return ['success' => false, 'error' => 'Authentication failed'];
        }

        // Get movie info
        $vodInfo = $api->getVODInfo($streamId);

        if (!$vodInfo || !isset($vodInfo['info'])) {
            return ['success' => false, 'error' => 'Movie not found'];
        }

        $movieData = array_merge(
            $vodInfo['info'],
            $vodInfo['movie_data'] ?? []
        );

        $movieData['stream_id'] = $streamId;

        $result = $this->processMovie($movieData, $server, $api, true);

        return [
            'success' => true,
            'action' => $result['action'],
            'movie_id' => $result['movie_id']
        ];
    }

    /**
     * Smart sync - only import new content
     */
    public function smartSync(int $serverId): array
    {
        $server = $this->serverManager->getServer($serverId);

        if (!$server) {
            return ['success' => false, 'error' => 'Server not found'];
        }

        $this->logger->info("Starting smart sync", ['server_id' => $serverId]);

        // Get existing stream IDs for this server
        $stmt = $this->db->prepare("
            SELECT stream_id FROM movie_links WHERE server_id = ?
        ");
        $stmt->execute([$serverId]);
        $existingStreamIds = $stmt->fetchAll(PDO::FETCH_COLUMN);

        // Get all movies from server
        $api = $this->serverManager->getServerAPI($serverId);

        if (!$api || !$api->authenticate()) {
            return ['success' => false, 'error' => 'Authentication failed'];
        }

        $allMovies = $api->getAllVODStreams(function ($current, $total, $category) {
            $this->logger->debug("Syncing category", [
                'current' => $current,
                'total' => $total,
                'category' => $category
            ]);
        });

        // Find new movies (not in our database)
        $newMovies = array_filter($allMovies, function ($movie) use ($existingStreamIds) {
            return !in_array($movie['stream_id'], $existingStreamIds);
        });

        $this->logger->info("Smart sync found new movies", [
            'total_on_server' => count($allMovies),
            'existing' => count($existingStreamIds),
            'new' => count($newMovies)
        ]);

        $result = [
            'success' => true,
            'total_on_server' => count($allMovies),
            'existing' => count($existingStreamIds),
            'new_found' => count($newMovies),
            'imported' => 0,
            'errors' => 0
        ];

        // Import new movies
        foreach ($newMovies as $movie) {
            try {
                $importResult = $this->processMovie($movie, $server, $api, false);
                if ($importResult['action'] === 'imported') {
                    $result['imported']++;
                }
            } catch (Exception $e) {
                $result['errors']++;
            }
        }

        // Update server last sync
        $this->serverManager->updateServer($serverId, [
            'last_sync' => date('Y-m-d H:i:s'),
            'movies_count' => count($allMovies)
        ]);

        $this->logger->info("Smart sync completed", $result);

        return $result;
    }

    /**
     * Check and select best link for a movie
     */
    public function selectBestLink(int $movieId): ?array
    {
        $stmt = $this->db->prepare("
            SELECT ml.*, s.name as server_name, s.priority as server_priority
            FROM movie_links ml
            JOIN servers s ON ml.server_id = s.id
            WHERE ml.movie_id = ? AND ml.is_working = 1 AND s.is_active = 1
            ORDER BY ml.is_preferred DESC, s.priority DESC, ml.check_response_time ASC
            LIMIT 1
        ");

        $stmt->execute([$movieId]);

        return $stmt->fetch() ?: null;
    }

    /**
     * Check all links for a movie
     */
    public function checkMovieLinks(int $movieId): array
    {
        $stmt = $this->db->prepare("
            SELECT ml.*, s.name as server_name
            FROM movie_links ml
            JOIN servers s ON ml.server_id = s.id
            WHERE ml.movie_id = ?
        ");

        $stmt->execute([$movieId]);
        $links = $stmt->fetchAll();

        $results = [];

        foreach ($links as $link) {
            $api = $this->serverManager->getServerAPI($link['server_id']);

            if ($api) {
                $checkResult = $api->checkStreamUrl($link['stream_url']);

                // Update link status
                $this->db->prepare("
                    UPDATE movie_links SET
                        is_working = :is_working,
                        last_check = NOW(),
                        check_response_time = :response_time,
                        error_count = CASE WHEN :is_working = 0 THEN error_count + 1 ELSE 0 END
                    WHERE id = :id
                ")->execute([
                    'id' => $link['id'],
                    'is_working' => $checkResult['is_working'] ? 1 : 0,
                    'response_time' => $checkResult['response_time']
                ]);

                $results[] = array_merge($link, $checkResult);
            }
        }

        return $results;
    }

    /**
     * Get newest movies (recently added)
     */
    public function getNewestMovies(int $limit = 50): array
    {
        $stmt = $this->db->prepare("
            SELECT m.*, COUNT(ml.id) as link_count
            FROM movies m
            LEFT JOIN movie_links ml ON m.id = ml.movie_id
            GROUP BY m.id
            ORDER BY m.created_at DESC
            LIMIT ?
        ");

        $stmt->execute([$limit]);

        return $stmt->fetchAll();
    }

    /**
     * Search movies
     */
    public function searchMovies(string $query, int $limit = 50): array
    {
        $stmt = $this->db->prepare("
            SELECT m.*, COUNT(ml.id) as link_count
            FROM movies m
            LEFT JOIN movie_links ml ON m.id = ml.movie_id
            WHERE MATCH(m.title, m.original_title, m.description) AGAINST (:query IN BOOLEAN MODE)
               OR m.title LIKE :like_query
            GROUP BY m.id
            ORDER BY m.rating DESC
            LIMIT :limit
        ");

        $stmt->bindValue('query', $query);
        $stmt->bindValue('like_query', "%{$query}%");
        $stmt->bindValue('limit', $limit, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll();
    }

    /**
     * Get movies by category
     */
    public function getMoviesByCategory(int $categoryId, int $limit = 100, int $offset = 0): array
    {
        $stmt = $this->db->prepare("
            SELECT DISTINCT m.*
            FROM movies m
            JOIN movie_links ml ON m.id = ml.movie_id
            WHERE ml.category_id = :category_id
            ORDER BY m.added_timestamp DESC
            LIMIT :limit OFFSET :offset
        ");

        $stmt->bindValue('category_id', $categoryId);
        $stmt->bindValue('limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue('offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll();
    }

    /**
     * Get import statistics
     */
    public function getImportStats(): array
    {
        $stats = [];

        // Total movies
        $stmt = $this->db->query("SELECT COUNT(*) FROM movies");
        $stats['total_movies'] = (int) $stmt->fetchColumn();

        // Movies by year
        $stmt = $this->db->query("
            SELECT year, COUNT(*) as count
            FROM movies
            WHERE year IS NOT NULL
            GROUP BY year
            ORDER BY year DESC
            LIMIT 10
        ");
        $stats['by_year'] = $stmt->fetchAll();

        // Total links
        $stmt = $this->db->query("SELECT COUNT(*) FROM movie_links");
        $stats['total_links'] = (int) $stmt->fetchColumn();

        // Working vs broken links
        $stmt = $this->db->query("
            SELECT
                SUM(CASE WHEN is_working = 1 THEN 1 ELSE 0 END) as working,
                SUM(CASE WHEN is_working = 0 THEN 1 ELSE 0 END) as broken
            FROM movie_links
        ");
        $linkStats = $stmt->fetch();
        $stats['working_links'] = (int) $linkStats['working'];
        $stats['broken_links'] = (int) $linkStats['broken'];

        // Movies per server
        $stmt = $this->db->query("
            SELECT s.name, COUNT(DISTINCT ml.movie_id) as movie_count
            FROM servers s
            LEFT JOIN movie_links ml ON s.id = ml.server_id
            GROUP BY s.id
            ORDER BY movie_count DESC
        ");
        $stats['by_server'] = $stmt->fetchAll();

        // Recent imports (last 24 hours)
        $stmt = $this->db->query("
            SELECT COUNT(*) FROM movies
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ");
        $stats['imported_24h'] = (int) $stmt->fetchColumn();

        return $stats;
    }
}
