# 🔧 دليل إصلاح مشكلة التوكن في التطبيق

## 🔴 المشكلة المُكتشفة

التطبيق **لا يستخدم** التوكن الذي يستلمه من السيرفر!

### التحليل:

```
📍 Step 1 - التفعيل (Activation):
   الطلب: {"code":"347827000488","mac":"...","mode":"active"}
   الاستجابة: {"token":"8f3485eea35823e67d56feced6e37003..."}
                         ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                         التوكن الصحيح من السيرفر ✅

📍 Step 2 - الطلب اللاحق (movies_latest):
   الطلب: {"token":"87f523c1bc192894af01df04a40df0f5..."}
                    ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
                    توكن مختلف تماماً! ❌

النتيجة: ❌ error: invalid or expired token
```

### 🔍 التفاصيل:

| | التوكن المُستلم | التوكن المُرسل |
|---|---|---|
| **MD5 Prefix** | `8f3485eea35823e67d56feced6e37003` | `87f523c1bc192894af01df04a40df0f5` |
| **أول 10 أحرف** | `8f3485eea3` | `87f523c1bc` |
| **في قاعدة البيانات** | ✅ موجود | ❌ غير موجود |
| **الحالة** | صحيح ويعمل | خاطئ ومرفوض |

---

## 🎯 الأسباب المحتملة

### 1️⃣ التطبيق يُنشئ توكن خاص به

```java
// ❌ خطأ: يُنشئ توكن جديد بدلاً من استخدام استجابة السيرفر
String token = MD5.hash(UUID.randomUUID().toString());
prefs.edit().putString("token", token).apply();
```

### 2️⃣ التطبيق يستخدم توكن قديم محفوظ

```java
// ❌ خطأ: يقرأ توكن قديم ولا يحدثه
String oldToken = prefs.getString("token", "");
if (!oldToken.isEmpty()) {
    // يستخدم التوكن القديم بدلاً من الجديد!
    useToken(oldToken);
}
```

### 3️⃣ التطبيق يحفظ في مكان خاطئ

```java
// ❌ خطأ: يحفظ في SharedPreferences مختلف
SharedPreferences prefs1 = getSharedPreferences("prefs1", MODE_PRIVATE);
String token = response.getString("token");
prefs1.edit().putString("token", token).apply();  // يحفظ هنا

// لكن يقرأ من مكان آخر!
SharedPreferences prefs2 = getSharedPreferences("prefs2", MODE_PRIVATE);
String savedToken = prefs2.getString("token", "");  // يقرأ من هنا!
```

### 4️⃣ التوكن يتم تعديله أثناء الحفظ/القراءة

```java
// ❌ خطأ: يعدل التوكن
String token = response.getString("token");
token = token.replace("/", "\\/");  // يضيف escape!
prefs.edit().putString("token", token).apply();
```

---

## ✅ الحل الكامل

### 📱 كود Android الصحيح

```java
/**
 * Token Manager - الطريقة الصحيحة
 */
public class TokenManager {
    private static final String TAG = "TokenManager";
    private static final String PREFS_NAME = "app_auth";
    private static final String KEY_TOKEN = "token";

    private SharedPreferences prefs;

    public TokenManager(Context context) {
        this.prefs = context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE);
    }

    /**
     * ═══════════════════════════════════════════════════════════════
     * ⭐ STEP 1: حفظ التوكن بعد التفعيل
     * ═══════════════════════════════════════════════════════════════
     */
    public void saveTokenFromActivation(JSONObject activationResponse) {
        try {
            // 1. استخرج التوكن من الاستجابة
            if (!activationResponse.has("token")) {
                Log.e(TAG, "❌ No token in activation response!");
                return;
            }

            String token = activationResponse.getString("token");

            // 2. Log التوكن المُستلم
            Log.d(TAG, "═══════════════════════════════════════════");
            Log.d(TAG, "Token RECEIVED from server:");
            Log.d(TAG, "Full: " + token);
            Log.d(TAG, "MD5:  " + token.substring(0, 32));
            Log.d(TAG, "Len:  " + token.length());
            Log.d(TAG, "═══════════════════════════════════════════");

            // 3. احفظ التوكن بالضبط كما استلمته - لا تعدله!
            prefs.edit()
                .putString(KEY_TOKEN, token)
                .apply();

            // 4. تحقق من الحفظ
            String savedToken = prefs.getString(KEY_TOKEN, "");

            Log.d(TAG, "Token SAVED to storage:");
            Log.d(TAG, "Full: " + savedToken);
            Log.d(TAG, "MD5:  " + savedToken.substring(0, 32));

            // 5. قارن
            if (token.equals(savedToken)) {
                Log.d(TAG, "✅ Token saved correctly!");
            } else {
                Log.e(TAG, "❌ CRITICAL ERROR: Token was MODIFIED during save!");
                Log.e(TAG, "Original MD5: " + token.substring(0, 32));
                Log.e(TAG, "Saved MD5:    " + savedToken.substring(0, 32));
            }

        } catch (JSONException e) {
            Log.e(TAG, "Error saving token", e);
        }
    }

    /**
     * ═══════════════════════════════════════════════════════════════
     * ⭐ STEP 2: استخدام التوكن في الطلبات
     * ═══════════════════════════════════════════════════════════════
     */
    public String getToken() {
        String token = prefs.getString(KEY_TOKEN, "");

        if (!token.isEmpty()) {
            Log.d(TAG, "Token loaded from storage:");
            Log.d(TAG, "MD5: " + token.substring(0, Math.min(32, token.length())));
        } else {
            Log.e(TAG, "❌ No token in storage!");
        }

        return token;
    }

    /**
     * ═══════════════════════════════════════════════════════════════
     * ⭐ STEP 3: إنشاء طلب مع التوكن
     * ═══════════════════════════════════════════════════════════════
     */
    public JSONObject createRequest(String mode) throws JSONException {
        String token = getToken();

        if (token.isEmpty()) {
            throw new IllegalStateException("No token available - activation required");
        }

        JSONObject request = new JSONObject();
        request.put("mode", mode);
        request.put("token", token);  // ⭐ استخدم التوكن بالضبط كما هو

        Log.d(TAG, "═══════════════════════════════════════════");
        Log.d(TAG, "Creating request:");
        Log.d(TAG, "Mode:  " + mode);
        Log.d(TAG, "Token MD5: " + token.substring(0, 32));
        Log.d(TAG, "═══════════════════════════════════════════");

        return request;
    }

    /**
     * حذف التوكن (للخروج)
     */
    public void clearToken() {
        prefs.edit().remove(KEY_TOKEN).apply();
        Log.d(TAG, "Token cleared");
    }
}
```

### 📋 كيفية الاستخدام

```java
/**
 * في Activity الرئيسي
 */
public class MainActivity extends AppCompatActivity {

    private TokenManager tokenManager;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        tokenManager = new TokenManager(this);

        // تحقق من وجود توكن
        if (tokenManager.getToken().isEmpty()) {
            // لا يوجد توكن - اذهب للتفعيل
            showActivationScreen();
        } else {
            // يوجد توكن - اذهب للصفحة الرئيسية
            loadMainContent();
        }
    }

    /**
     * عند التفعيل
     */
    private void activateCode(String code, String mac) {
        ApiClient.activate(code, mac, new ApiCallback() {
            @Override
            public void onSuccess(String encryptedResponse) {
                // فك التشفير
                String decrypted = XOREncryption.decrypt(encryptedResponse, XOR_KEY);

                try {
                    JSONObject response = new JSONObject(decrypted);

                    if (response.getInt("status") == 100) {
                        // ⭐ احفظ التوكن
                        tokenManager.saveTokenFromActivation(response);

                        // اذهب للصفحة الرئيسية
                        loadMainContent();
                    } else {
                        showError(response.getString("message"));
                    }

                } catch (JSONException e) {
                    showError("Invalid response");
                }
            }

            @Override
            public void onError(String error) {
                showError(error);
            }
        });
    }

    /**
     * عند طلب بيانات
     */
    private void loadMovies() {
        try {
            // ⭐ أنشئ طلب مع التوكن
            JSONObject request = tokenManager.createRequest("movies_latest");

            // شفّر وأرسل
            String encrypted = XOREncryption.encrypt(request.toString(), XOR_KEY);

            ApiClient.send(encrypted, new ApiCallback() {
                @Override
                public void onSuccess(String encryptedResponse) {
                    String decrypted = XOREncryption.decrypt(encryptedResponse, XOR_KEY);

                    // تحقق من الأخطاء
                    if (decrypted.contains("invalid or expired token")) {
                        Log.e("MainActivity", "❌ Token is invalid!");
                        tokenManager.clearToken();
                        showActivationScreen();
                    } else {
                        // معالجة البيانات
                        displayMovies(decrypted);
                    }
                }

                @Override
                public void onError(String error) {
                    showError(error);
                }
            });

        } catch (Exception e) {
            showError(e.getMessage());
        }
    }
}
```

---

## 🧪 خطوات التشخيص

### 1️⃣ أضف Logging مكثف

```java
// بعد التفعيل مباشرة
Log.d("TOKEN", "════════════════════════════════");
Log.d("TOKEN", "STEP 1: Activation Response");
Log.d("TOKEN", "Received: " + token);
Log.d("TOKEN", "MD5: " + token.substring(0, 32));
Log.d("TOKEN", "════════════════════════════════");

// عند الحفظ
Log.d("TOKEN", "STEP 2: Saving Token");
prefs.edit().putString("token", token).apply();
Log.d("TOKEN", "Saved!");

// عند القراءة
Log.d("TOKEN", "STEP 3: Reading Token");
String saved = prefs.getString("token", "");
Log.d("TOKEN", "Loaded: " + saved);
Log.d("TOKEN", "MD5: " + saved.substring(0, 32));
Log.d("TOKEN", "════════════════════════════════");

// عند الإرسال
Log.d("TOKEN", "STEP 4: Sending Request");
Log.d("TOKEN", "Using: " + tokenToSend);
Log.d("TOKEN", "MD5: " + tokenToSend.substring(0, 32));
Log.d("TOKEN", "════════════════════════════════");
```

### 2️⃣ راقب logcat

```bash
adb logcat | grep TOKEN
```

يجب أن ترى **نفس MD5** في جميع المراحل!

### 3️⃣ قارن القيم

```
✅ الصحيح:
STEP 1: Received MD5: 8f3485eea35823e67d56feced6e37003
STEP 2: Saved MD5:    8f3485eea35823e67d56feced6e37003
STEP 3: Loaded MD5:   8f3485eea35823e67d56feced6e37003
STEP 4: Sending MD5:  8f3485eea35823e67d56feced6e37003

❌ الخطأ الحالي:
STEP 1: Received MD5: 8f3485eea35823e67d56feced6e37003
STEP 2: Saved MD5:    8f3485eea35823e67d56feced6e37003
STEP 3: Loaded MD5:   8f3485eea35823e67d56feced6e37003
STEP 4: Sending MD5:  87f523c1bc192894af01df04a40df0f5  ← من أين جاء هذا؟!
```

---

## ✅ التحقق من الإصلاح

بعد تطبيق التعديلات:

```bash
# 1. فعّل كود جديد في التطبيق
# 2. راقب logcat وسجل MD5
# 3. أرسل طلب movies_latest
# 4. تحقق أن MD5 نفسه في جميع المراحل

# من السيرفر:
php /var/www/html/iptv/test_token.php "token_from_app"
# يجب أن يكون: ✅ SUCCESS!
```

---

## 📊 الملخص

| الخطوة | الحالي (❌ خطأ) | المطلوب (✅ صحيح) |
|--------|----------------|-------------------|
| **التفعيل** | يستلم التوكن | يستلم التوكن |
| **الحفظ** | يحفظ في مكان ما | يحفظ في KEY_TOKEN |
| **القراءة** | يقرأ توكن مختلف! | يقرأ نفس التوكن |
| **الإرسال** | يرسل توكن خاطئ | يرسل التوكن الصحيح |
| **النتيجة** | ❌ invalid token | ✅ SUCCESS |

---

**المفتاح:** استخدم **نفس التوكن** الذي تستلمه من السيرفر، **لا تعدله**، **لا تُنشئ توكن جديد**!

---

**آخر تحديث:** 2025-11-07
