/**
 * ═══════════════════════════════════════════════════════════════════════════
 * Android Token Management - Complete Example
 * ═══════════════════════════════════════════════════════════════════════════
 *
 * This example shows how to:
 * 1. Check if token exists
 * 2. Activate code and get token if empty
 * 3. Save token properly
 * 4. Use token for API requests
 */

import android.content.Context;
import android.content.SharedPreferences;
import android.util.Log;
import org.json.JSONObject;
import okhttp3.*;
import java.io.IOException;

public class TokenManager {

    private static final String TAG = "TokenManager";
    private static final String PREFS_NAME = "app_data";
    private static final String KEY_TOKEN = "token";
    private static final String KEY_USERNAME = "username";
    private static final String KEY_PASSWORD = "password";
    private static final String KEY_EXPIRE = "expire";
    private static final String KEY_IS_ACTIVATED = "is_activated";

    private static final String API_URL = "http://45.154.207.45/iptv/V6APK/API-V6APK.php";
    private static final String XOR_KEY = "KvuZRm365wk5Sz&Xy.Zx";

    private Context context;
    private SharedPreferences prefs;
    private OkHttpClient client;

    public TokenManager(Context context) {
        this.context = context;
        this.prefs = context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE);
        this.client = new OkHttpClient();
    }

    /**
     * ═══════════════════════════════════════════════════════════════════════
     * STEP 1: Check if Token Exists
     * ═══════════════════════════════════════════════════════════════════════
     */
    public boolean hasToken() {
        String token = prefs.getString(KEY_TOKEN, "");
        boolean hasToken = !token.isEmpty();

        Log.d(TAG, "Token exists: " + hasToken);
        if (hasToken) {
            Log.d(TAG, "Token preview: " + token.substring(0, Math.min(32, token.length())) + "...");
        }

        return hasToken;
    }

    /**
     * ═══════════════════════════════════════════════════════════════════════
     * STEP 2: Activate Code (When Token is Empty)
     * ═══════════════════════════════════════════════════════════════════════
     */
    public void activateCode(String code, String mac, ActivationCallback callback) {
        Log.d(TAG, "Starting activation for code: " + code);

        try {
            // Create activation request
            JSONObject request = new JSONObject();
            request.put("code", code);
            request.put("mac", mac);
            request.put("sn", mac);
            request.put("model", android.os.Build.MODEL);
            request.put("group", 1);
            request.put("mode", "active");

            Log.d(TAG, "Activation request: " + request.toString());

            // Encrypt request
            String encrypted = XOREncryption.encrypt(request.toString(), XOR_KEY);

            // Send request
            RequestBody body = new FormBody.Builder()
                .add("json", encrypted)
                .build();

            Request httpRequest = new Request.Builder()
                .url(API_URL)
                .post(body)
                .addHeader("User-Agent", "okhttp/3.10.0")
                .addHeader("Content-Type", "application/x-www-form-urlencoded")
                .build();

            client.newCall(httpRequest).enqueue(new Callback() {
                @Override
                public void onFailure(Call call, IOException e) {
                    Log.e(TAG, "Activation request failed", e);
                    callback.onError("Network error: " + e.getMessage());
                }

                @Override
                public void onResponse(Call call, Response response) throws IOException {
                    if (!response.isSuccessful()) {
                        callback.onError("Server error: " + response.code());
                        return;
                    }

                    // Decrypt response
                    String encryptedResponse = response.body().string();
                    String decrypted = XOREncryption.decrypt(encryptedResponse, XOR_KEY);

                    Log.d(TAG, "Activation response: " + decrypted);

                    try {
                        JSONObject activationResponse = new JSONObject(decrypted);
                        processActivationResponse(activationResponse, callback);
                    } catch (Exception e) {
                        Log.e(TAG, "Failed to parse activation response", e);
                        callback.onError("Invalid response format");
                    }
                }
            });

        } catch (Exception e) {
            Log.e(TAG, "Failed to create activation request", e);
            callback.onError("Failed to create request: " + e.getMessage());
        }
    }

    /**
     * ═══════════════════════════════════════════════════════════════════════
     * STEP 3: Process Activation Response and Save Token
     * ═══════════════════════════════════════════════════════════════════════
     */
    private void processActivationResponse(JSONObject response, ActivationCallback callback) {
        try {
            int status = response.getInt("status");
            String message = response.getString("message");

            Log.d(TAG, "Activation status: " + status + " - " + message);

            if (status != 100) {
                callback.onError("Activation failed: " + message);
                return;
            }

            // ⭐ IMPORTANT: Extract token from response
            if (!response.has("token")) {
                callback.onError("No token in response!");
                return;
            }

            String token = response.getString("token");
            String username = response.getString("username");
            String password = response.getString("password");
            String expire = response.getString("expire");

            // ⭐ CRITICAL: Save token EXACTLY as received - NO MODIFICATIONS!
            Log.d(TAG, "Saving token...");
            Log.d(TAG, "Token length: " + token.length());
            Log.d(TAG, "Token preview: " + token.substring(0, Math.min(50, token.length())) + "...");

            boolean saved = saveToken(token, username, password, expire);

            if (!saved) {
                callback.onError("Failed to save token");
                return;
            }

            // Verify token was saved correctly
            String savedToken = getToken();
            if (!savedToken.equals(token)) {
                Log.e(TAG, "⚠️ WARNING: Saved token doesn't match!");
                Log.e(TAG, "Original: " + token.substring(0, 32));
                Log.e(TAG, "Saved:    " + savedToken.substring(0, 32));
                callback.onError("Token save verification failed");
                return;
            }

            Log.d(TAG, "✅ Token saved and verified successfully!");

            // Return success with account info
            AccountInfo accountInfo = new AccountInfo();
            accountInfo.username = username;
            accountInfo.password = password;
            accountInfo.expire = expire;
            accountInfo.token = token;
            accountInfo.message = message;

            callback.onSuccess(accountInfo);

        } catch (Exception e) {
            Log.e(TAG, "Failed to process activation response", e);
            callback.onError("Processing error: " + e.getMessage());
        }
    }

    /**
     * ═══════════════════════════════════════════════════════════════════════
     * Save Token to SharedPreferences
     * ═══════════════════════════════════════════════════════════════════════
     */
    private boolean saveToken(String token, String username, String password, String expire) {
        try {
            SharedPreferences.Editor editor = prefs.edit();

            // ⭐ Save token EXACTLY as is - don't modify it!
            editor.putString(KEY_TOKEN, token);
            editor.putString(KEY_USERNAME, username);
            editor.putString(KEY_PASSWORD, password);
            editor.putString(KEY_EXPIRE, expire);
            editor.putBoolean(KEY_IS_ACTIVATED, true);

            // Use apply() for async save
            editor.apply();

            Log.d(TAG, "Token and account info saved to SharedPreferences");
            return true;

        } catch (Exception e) {
            Log.e(TAG, "Failed to save token", e);
            return false;
        }
    }

    /**
     * ═══════════════════════════════════════════════════════════════════════
     * Get Token from SharedPreferences
     * ═══════════════════════════════════════════════════════════════════════
     */
    public String getToken() {
        return prefs.getString(KEY_TOKEN, "");
    }

    /**
     * ═══════════════════════════════════════════════════════════════════════
     * STEP 4: Use Token for API Requests
     * ═══════════════════════════════════════════════════════════════════════
     */
    public void makeRequest(String mode, ApiCallback callback) {
        String token = getToken();

        if (token.isEmpty()) {
            Log.e(TAG, "❌ No token available! User must activate first.");
            callback.onError("No token - activation required");
            return;
        }

        Log.d(TAG, "Making request with mode: " + mode);
        Log.d(TAG, "Using token: " + token.substring(0, 32) + "...");

        try {
            // Create request with token
            JSONObject request = new JSONObject();
            request.put("mode", mode);
            request.put("token", token); // ⭐ Use token as is - don't modify!

            // Encrypt request
            String encrypted = XOREncryption.encrypt(request.toString(), XOR_KEY);

            // Send request
            RequestBody body = new FormBody.Builder()
                .add("json", encrypted)
                .build();

            Request httpRequest = new Request.Builder()
                .url(API_URL)
                .post(body)
                .addHeader("User-Agent", "okhttp/3.10.0")
                .build();

            client.newCall(httpRequest).enqueue(new Callback() {
                @Override
                public void onFailure(Call call, IOException e) {
                    Log.e(TAG, "Request failed", e);
                    callback.onError("Network error: " + e.getMessage());
                }

                @Override
                public void onResponse(Call call, Response response) throws IOException {
                    String encryptedResponse = response.body().string();
                    String decrypted = XOREncryption.decrypt(encryptedResponse, XOR_KEY);

                    // Check for errors
                    if (decrypted.contains("invalid or expired token")) {
                        Log.e(TAG, "❌ Token is invalid or expired!");
                        Log.e(TAG, "Token used: " + token.substring(0, 50) + "...");
                        callback.onError("invalid or expired token");
                        return;
                    }

                    if (decrypted.startsWith("error:")) {
                        callback.onError(decrypted);
                        return;
                    }

                    Log.d(TAG, "✅ Request successful!");
                    callback.onSuccess(decrypted);
                }
            });

        } catch (Exception e) {
            Log.e(TAG, "Failed to make request", e);
            callback.onError("Request error: " + e.getMessage());
        }
    }

    /**
     * Clear all saved data (logout)
     */
    public void clearToken() {
        prefs.edit().clear().apply();
        Log.d(TAG, "Token and all data cleared");
    }

    // ═══════════════════════════════════════════════════════════════════════
    // Callbacks
    // ═══════════════════════════════════════════════════════════════════════

    public interface ActivationCallback {
        void onSuccess(AccountInfo accountInfo);
        void onError(String error);
    }

    public interface ApiCallback {
        void onSuccess(String response);
        void onError(String error);
    }

    public static class AccountInfo {
        public String username;
        public String password;
        public String expire;
        public String token;
        public String message;
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════════════
 * USAGE EXAMPLE IN ACTIVITY
 * ═══════════════════════════════════════════════════════════════════════════
 */
class ExampleActivity extends Activity {

    private TokenManager tokenManager;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        tokenManager = new TokenManager(this);

        // Check if we have a token
        if (!tokenManager.hasToken()) {
            // Token is empty - need to activate
            Log.d("Example", "No token found - showing activation screen");
            showActivationScreen();
        } else {
            // Token exists - proceed to main app
            Log.d("Example", "Token found - proceeding to main app");
            loadMainContent();
        }
    }

    private void showActivationScreen() {
        // Show UI for user to enter activation code
        // When user submits code:
        String code = "225666426500";
        String mac = "8c:fd:f0:00:7f:13";

        tokenManager.activateCode(code, mac, new TokenManager.ActivationCallback() {
            @Override
            public void onSuccess(TokenManager.AccountInfo info) {
                runOnUiThread(() -> {
                    Log.d("Example", "✅ Activation successful!");
                    Log.d("Example", "Username: " + info.username);
                    Log.d("Example", "Expires: " + info.expire);

                    // Now we have token - load main content
                    loadMainContent();
                });
            }

            @Override
            public void onError(String error) {
                runOnUiThread(() -> {
                    Log.e("Example", "❌ Activation failed: " + error);
                    Toast.makeText(ExampleActivity.this,
                        "Activation failed: " + error,
                        Toast.LENGTH_LONG).show();
                });
            }
        });
    }

    private void loadMainContent() {
        // Use token to fetch data
        tokenManager.makeRequest("movies_latest", new TokenManager.ApiCallback() {
            @Override
            public void onSuccess(String response) {
                Log.d("Example", "✅ Got movies data!");
                // Parse and display movies
            }

            @Override
            public void onError(String error) {
                Log.e("Example", "❌ Request failed: " + error);

                if (error.contains("invalid or expired token")) {
                    // Token expired - need to reactivate
                    tokenManager.clearToken();
                    showActivationScreen();
                }
            }
        });
    }
}
