# 🔐 حل مشكلة التوكن الفارغ (Token Empty Solution)

## 📊 التدفق الكامل (Flow Chart)

```
┌─────────────────────────────────────────────────────────────────┐
│                    APP STARTS                                   │
└────────────────────────────┬────────────────────────────────────┘
                             │
                             ▼
                   ┌──────────────────┐
                   │  Check Token     │
                   │  hasToken()?     │
                   └────────┬─────────┘
                            │
                ┌───────────┴───────────┐
                │                       │
                ▼ NO                    ▼ YES
       ┌─────────────────┐      ┌──────────────────┐
       │  Token EMPTY    │      │  Token EXISTS    │
       │                 │      │                  │
       │  NEED TO        │      │  GO TO MAIN APP  │
       │  ACTIVATE       │      │  Use Token       │
       └────────┬────────┘      └──────────────────┘
                │
                ▼
       ┌─────────────────────┐
       │ Show Activation UI  │
       │ - Enter Code        │
       │ - Enter MAC         │
       └────────┬────────────┘
                │
                ▼
       ┌─────────────────────┐
       │ Send Activation     │
       │ mode: "active"      │
       │ code: "xxx"         │
       └────────┬────────────┘
                │
                ▼
       ┌─────────────────────┐
       │ Receive Response    │
       │ status: 100?        │
       └────────┬────────────┘
                │
        ┌───────┴────────┐
        │                │
        ▼ YES            ▼ NO
┌───────────────┐  ┌────────────────┐
│ Extract Token │  │  Show Error    │
│ Save Token    │  │  Try Again     │
│ Save Username │  └────────────────┘
│ Save Password │
└───────┬───────┘
        │
        ▼
┌───────────────────┐
│ Verify Save       │
│ savedToken ==     │
│ receivedToken?    │
└───────┬───────────┘
        │
        ▼ YES
┌───────────────────┐
│ GO TO MAIN APP    │
│ Use Token for     │
│ All Requests      │
└───────────────────┘
```

---

## 🎯 الحل السريع (Quick Solution)

### 1️⃣ إذا كان التوكن فارغاً:

```java
if (!tokenManager.hasToken()) {
    // الحل: فعّل الكود للحصول على توكن
    tokenManager.activateCode(code, mac, new ActivationCallback() {
        @Override
        public void onSuccess(AccountInfo info) {
            // ✅ التوكن تم حفظه تلقائياً
            // الآن يمكنك استخدام التطبيق
            startMainActivity();
        }

        @Override
        public void onError(String error) {
            // ❌ التفعيل فشل
            showError(error);
        }
    });
}
```

### 2️⃣ طلب التفعيل (Activation Request):

```json
{
    "code": "225666426500",
    "mac": "8c:fd:f0:00:7f:13",
    "sn": "8c:fd:f0:00:7f:13",
    "model": "SM-N975F",
    "group": 1,
    "mode": "active"
}
```

### 3️⃣ استجابة التفعيل (Activation Response):

```json
{
    "status": 100,
    "message": "The Code is active",
    "username": "225666426500",
    "password": "545229799690e1282accf6",
    "expire": "2025-11-14",
    "token": "7682126a6f006d9aafeebdb81a192cc5zilRcL7F7CnZTs7m09PV5SH1xUfcpWTsJgMpYVwCklaTFVn2lPMAnTq5ra/DprlwB6lYZedqNjuwZlQwIER0KA==",
    ...
}
```

### 4️⃣ حفظ التوكن:

```java
// ⭐ مهم جداً: احفظ التوكن كما هو بدون تعديل!
String token = response.getString("token");

SharedPreferences prefs = getSharedPreferences("app_data", MODE_PRIVATE);
prefs.edit()
    .putString("token", token)           // ⭐ بالضبط كما استلمته
    .putString("username", username)
    .putString("password", password)
    .putString("expire", expire)
    .putBoolean("is_activated", true)
    .apply();

// تحقق من الحفظ
String saved = prefs.getString("token", "");
if (!saved.equals(token)) {
    Log.e("TOKEN", "⚠️ WARNING: Token was modified during save!");
}
```

### 5️⃣ استخدام التوكن:

```java
// في جميع الطلبات بعد التفعيل
String token = prefs.getString("token", "");

JSONObject request = new JSONObject();
request.put("mode", "movies_latest");
request.put("token", token);  // ⭐ استخدمه مباشرة

// شفّر وأرسل
String encrypted = XOREncryption.encrypt(request.toString(), XOR_KEY);
sendRequest(encrypted);
```

---

## ⚠️ الأخطاء الشائعة

### ❌ خطأ 1: عدم التحقق من وجود التوكن

```java
// خطأ
String token = prefs.getString("token", "");
makeRequest(mode, token); // قد يكون فارغاً!

// صحيح
if (!tokenManager.hasToken()) {
    showActivationScreen();
    return;
}
tokenManager.makeRequest(mode, callback);
```

### ❌ خطأ 2: تعديل التوكن

```java
// خطأ - يضيف escape characters
String json = new Gson().toJson(request);

// خطأ - يعدل الشرطة المائلة
String token = response.getString("token").replace("/", "\\/");

// صحيح - لا تعديل
String token = response.getString("token");
```

### ❌ خطأ 3: عدم حفظ التوكن من الاستجابة

```java
// خطأ - يولد توكن جديد بدلاً من استخدام استجابة السيرفر
String token = generateToken(); // ❌

// صحيح - يستخدم التوكن من السيرفر
String token = response.getString("token"); // ✅
```

---

## 🧪 اختبار التوكن

### من السيرفر:

```bash
# اختبر أي توكن
php /var/www/html/iptv/test_token.php "your_token_here"

# فعّل كود جديد
php /var/www/html/iptv/activate_and_get_token.php <code> <mac>

# أضف توكن يدوياً
php /var/www/html/iptv/add_token_manually.php <username> "<token>"
```

### من التطبيق:

```java
// أضف logging
Log.d("TOKEN", "Received: " + receivedToken);
Log.d("TOKEN", "Saved: " + prefs.getString("token", ""));
Log.d("TOKEN", "Sending: " + tokenInRequest);

// قارن القيم في logcat
```

---

## 📝 نموذج التطبيق الكامل

راجع ملف: `AndroidTokenExample.java`

يحتوي على:
- ✅ التحقق من وجود التوكن
- ✅ التفعيل عند الفراغ
- ✅ الحفظ الصحيح
- ✅ الاستخدام في الطلبات
- ✅ معالجة الأخطاء
- ✅ مثال كامل للـ Activity

---

## 🎯 التوكن الحالي للكود 225666426500

```
7682126a6f006d9aafeebdb81a192cc5zilRcL7F7CnZTs7m09PV5SH1xUfcpWTsJgMpYVwCklaTFVn2lPMAnTq5ra/DprlwB6lYZedqNjuwZlQwIER0KA==
```

- ✅ **Username:** 225666426500
- ✅ **Password:** 545229799690e1282accf6
- ✅ **Expires:** 2025-11-14
- ✅ **Status:** Active and Working

---

## 📚 ملفات مفيدة

| الملف | الوصف |
|-------|--------|
| `activate_and_get_token.php` | تفعيل كود والحصول على توكن |
| `test_token.php` | اختبار صلاحية التوكن |
| `add_token_manually.php` | إضافة توكن يدوياً |
| `AndroidTokenExample.java` | مثال كامل لـ Android |
| `TOKEN_USAGE_GUIDE.md` | الدليل الكامل |
| `TOKEN_EMPTY_SOLUTION.md` | هذا الملف |

---

## ✅ خلاصة

**عندما يكون التوكن فارغاً:**

1. أرسل طلب تفعيل (`mode: "active"`)
2. استلم الاستجابة واستخرج `token`
3. احفظ التوكن **بالضبط كما هو**
4. تحقق من الحفظ
5. استخدم التوكن في جميع الطلبات

**القاعدة الذهبية:**
> 🔑 **لا تعدّل التوكن أبداً!**
> احفظه واستخدمه كما استلمته من السيرفر.

---

**آخر تحديث:** 2025-11-07
**الإصدار:** 1.0
