<?php
/**
 * Token Mismatch Diagnostic
 * Analyzing why app sends different token
 */

echo "╔═══════════════════════════════════════════════════════════════════════╗\n";
echo "║              🔍 TOKEN MISMATCH ANALYSIS                               ║\n";
echo "╚═══════════════════════════════════════════════════════════════════════╝\n\n";

// Token received from activation
$received_token = "8f3485eea35823e67d56feced6e370039w8ak8oN9bhC37wXmUsUfX5n2P/46vvsRwerMC84vuJndHXNWT3yuJz3V6nOYhQrHewijEGI+K/UPev4tVR/Ig==";

// Token sent by app in subsequent requests
$sent_token = "87f523c1bc192894af01df04a40df0f53Uex+ZvrGjkwkFmnhaXX68p56C+RTK6NY90Jco7uMld4qLnzyqfgMFP/zv3OmpVZ5R4YV2rZ5CKhqT/pmJnAwA==";

echo "📊 TOKEN COMPARISON\n";
echo str_repeat("─", 71) . "\n\n";

echo "1️⃣  Token RECEIVED from activation (Step 1):\n";
echo "   Code: 347827000488\n";
echo "   Full: $received_token\n";
echo "   MD5 Prefix: " . substr($received_token, 0, 32) . "\n";
echo "   Length: " . strlen($received_token) . " chars\n";
echo "   First 50: " . substr($received_token, 0, 50) . "...\n\n";

echo "2️⃣  Token SENT by app (Step 2 - movies_latest):\n";
echo "   Full: $sent_token\n";
echo "   MD5 Prefix: " . substr($sent_token, 0, 32) . "\n";
echo "   Length: " . strlen($sent_token) . " chars\n";
echo "   First 50: " . substr($sent_token, 0, 50) . "...\n\n";

echo "🔍 ANALYSIS:\n";
echo str_repeat("─", 71) . "\n\n";

// Compare MD5 prefixes
$received_md5 = substr($received_token, 0, 32);
$sent_md5 = substr($sent_token, 0, 32);

if ($received_md5 === $sent_md5) {
    echo "✅ MD5 Prefixes MATCH\n";
} else {
    echo "❌ MD5 Prefixes DO NOT MATCH!\n";
    echo "   Received: $received_md5\n";
    echo "   Sent:     $sent_md5\n";
    echo "   Diff:     ";
    for ($i = 0; $i < 32; $i++) {
        echo ($received_md5[$i] === $sent_md5[$i]) ? ' ' : '^';
    }
    echo "\n\n";
}

// First character comparison
echo "📍 First Character Comparison:\n";
echo "   Received: '" . $received_token[0] . "' (ASCII: " . ord($received_token[0]) . ")\n";
echo "   Sent:     '" . $sent_token[0] . "' (ASCII: " . ord($sent_token[0]) . ")\n";

if ($received_token[0] !== $sent_token[0]) {
    echo "   ❌ DIFFERENT from first character!\n\n";
}

// Check if tokens match
if ($received_token === $sent_token) {
    echo "✅ Tokens are IDENTICAL\n";
} else {
    echo "❌ Tokens are COMPLETELY DIFFERENT\n\n";
}

echo "🔬 DETAILED ANALYSIS:\n";
echo str_repeat("─", 71) . "\n\n";

// Find first difference
for ($i = 0; $i < min(strlen($received_token), strlen($sent_token)); $i++) {
    if ($received_token[$i] !== $sent_token[$i]) {
        echo "First difference at position $i:\n";
        echo "   Received[{$i}]: '{$received_token[$i]}' (ASCII " . ord($received_token[$i]) . ")\n";
        echo "   Sent[{$i}]:     '{$sent_token[$i]}' (ASCII " . ord($sent_token[$i]) . ")\n\n";

        echo "Context (10 chars before and after):\n";
        $start = max(0, $i - 10);
        $end = min(strlen($received_token), $i + 10);
        echo "   Received: " . substr($received_token, $start, $end - $start) . "\n";
        echo "   Sent:     " . substr($sent_token, $start, $end - $start) . "\n";
        break;
    }
}

echo "\n";
echo "💡 CONCLUSION:\n";
echo str_repeat("─", 71) . "\n\n";

echo "The app is NOT using the token it received!\n\n";

echo "Possible causes:\n";
echo "1. App generates its own token instead of using server's response\n";
echo "2. App has cached/old token from previous session\n";
echo "3. Token storage is overwritten by another process\n";
echo "4. Token is modified during save/load (encoding issue)\n";
echo "5. App is using wrong SharedPreferences key\n\n";

// Check database
require_once('/var/www/html/iptv/includes/Config.php');
$link = mysqli_connect(
    $config['db']['hostname'],
    $config['db']['username'],
    $config['db']['password'],
    $config['db']['database']
);

if ($link) {
    echo "🗄️  DATABASE CHECK:\n";
    echo str_repeat("─", 71) . "\n\n";

    // Check received token
    $safe_received = mysqli_real_escape_string($link, $received_token);
    $result1 = mysqli_query($link, "SELECT user_id, FROM_UNIXTIME(created) as created FROM user_tokens WHERE token = '$safe_received'");

    if ($result1 && mysqli_num_rows($result1) > 0) {
        $row = mysqli_fetch_assoc($result1);
        echo "✅ RECEIVED token EXISTS in database\n";
        echo "   User ID: " . $row['user_id'] . "\n";
        echo "   Created: " . $row['created'] . "\n\n";
    } else {
        echo "❌ RECEIVED token NOT FOUND in database\n\n";
    }

    // Check sent token
    $safe_sent = mysqli_real_escape_string($link, $sent_token);
    $result2 = mysqli_query($link, "SELECT user_id FROM user_tokens WHERE token = '$safe_sent'");

    if ($result2 && mysqli_num_rows($result2) > 0) {
        echo "⚠️  SENT token EXISTS in database (unexpected!)\n\n";
    } else {
        echo "❌ SENT token NOT FOUND in database (as expected)\n\n";
    }

    // Check for code 347827000488
    $result3 = mysqli_query($link, "
        SELECT u.id, u.username, t.token
        FROM users u
        LEFT JOIN user_tokens t ON u.id = t.user_id
        WHERE u.username = '347827000488'
        ORDER BY t.created DESC
        LIMIT 1
    ");

    if ($result3 && mysqli_num_rows($result3) > 0) {
        $user_data = mysqli_fetch_assoc($result3);
        echo "📌 Latest token for user 347827000488:\n";
        if ($user_data['token']) {
            echo "   Token: " . substr($user_data['token'], 0, 50) . "...\n";
            echo "   MD5:   " . substr($user_data['token'], 0, 32) . "\n\n";

            if ($user_data['token'] === $received_token) {
                echo "   ✅ Matches RECEIVED token\n";
            } else if ($user_data['token'] === $sent_token) {
                echo "   ⚠️  Matches SENT token\n";
            } else {
                echo "   ❌ Matches neither token!\n";
            }
        } else {
            echo "   ⚠️  No token found for this user\n";
        }
    }

    mysqli_close($link);
}

echo "\n";
echo "╔═══════════════════════════════════════════════════════════════════════╗\n";
echo "║                    🔧 SOLUTION                                        ║\n";
echo "╚═══════════════════════════════════════════════════════════════════════╝\n\n";

echo "In Android App - After activation:\n\n";
echo "❌ WRONG (What app is doing now):\n";
echo "-----------------------------------\n";
echo "String token = generateToken();  // Generating own token!\n";
echo "prefs.putString(\"token\", token);\n\n";

echo "✅ CORRECT (What app should do):\n";
echo "-----------------------------------\n";
echo "// Get token from server response\n";
echo "JSONObject response = new JSONObject(decryptedData);\n";
echo "String token = response.getString(\"token\");\n\n";

echo "// Log it immediately\n";
echo "Log.d(\"TOKEN\", \"Received from server: \" + token.substring(0, 32));\n\n";

echo "// Save it EXACTLY as received\n";
echo "prefs.edit().putString(\"token\", token).apply();\n\n";

echo "// Verify save\n";
echo "String saved = prefs.getString(\"token\", \"\");\n";
echo "Log.d(\"TOKEN\", \"Saved to storage: \" + saved.substring(0, 32));\n\n";

echo "// Verify they match\n";
echo "if (!saved.equals(token)) {\n";
echo "    Log.e(\"TOKEN\", \"❌ MISMATCH! Token was modified!\");\n";
echo "} else {\n";
echo "    Log.d(\"TOKEN\", \"✅ Token saved correctly\");\n";
echo "}\n\n";

echo "╔═══════════════════════════════════════════════════════════════════════╗\n";
echo "║  Add this logging to your app and check logcat to see where the      ║\n";
echo "║  token gets changed!                                                  ║\n";
echo "╚═══════════════════════════════════════════════════════════════════════╝\n";
