<?php 
define('_MM_MAX_INT_BYTES', (log(PHP_INT_MAX, 2) - 1) / 8);
class InvalidDatabaseException
{
    public function __construct($msg)
    {
        return $msg;
    }
}
class InvalidArgum
{
    public function __construct($msg)
    {
        return $msg;
    }
}
class Maxmind
{
    private static $DATA_SECTION_SEPARATOR_SIZE = 16;
    private static $METADATA_START_MARKER = "\xAB\xCD\xEF\x4D\x61\x78\x4D\x69\x6E\x64\x2E\x63\x6F\x6D";
    private static $METADATA_START_MARKER_LENGTH = 14;
    private static $METADATA_MAX_SIZE = 131072;
    private $decoder = null;
    private $fileHandle = null;
    private $fileSize = null;
    private $ipV4Start = null;
    private $metadata = null;
    public function __construct($database)
    {
        if( func_num_args() !== 1 ) 
        {
            new InvalidArgum('The constructor takes exactly one argument.');
        }
        if( !is_readable($database) ) 
        {
            new InvalidArgum('The file "' . $database . '" does not exist or is not readable.');
        }
        $this->fileHandle = @fopen($database, 'rb');
        if( $this->fileHandle === false ) 
        {
            new InvalidArgum('Error opening "' . $database . '".');
        }
        $this->fileSize = @filesize($database);
        if( $this->fileSize === false ) 
        {
            new InvalidArgum('Error determining the size of "' . $database . '".');
        }
        $start = $this->findMetadataStart($database);
        $metadataDecoder = new Decoder($this->fileHandle, $start);
        list($metadataArray) = $metadataDecoder->decode($start);
        $this->metadata = new Metadata($metadataArray);
        $this->decoder = new Decoder($this->fileHandle, $this->metadata->searchTreeSize + self::$DATA_SECTION_SEPARATOR_SIZE);
        $this->ipV4Start = $this->ipV4StartNode();
    }
    public function get($ipAddress)
    {
        if( func_num_args() !== 1 ) 
        {
            new InvalidArgum('Method takes exactly one argument.');
        }
        list($record) = $this->getWithPrefixLen($ipAddress);
        return $record;
    }
    public function getWithPrefixLen($ipAddress)
    {
        if( func_num_args() !== 1 ) 
        {
            new InvalidArgum('Method takes exactly one argument.');
        }
        if( !is_resource($this->fileHandle) ) 
        {
            new InvalidArgum('Attempt to read from a closed MaxMind DB.');
        }
        if( !filter_var($ipAddress, FILTER_VALIDATE_IP) ) 
        {
            new InvalidArgum('The value "' . $ipAddress . '" is not a valid IP address.');
        }
        list($pointer, $prefixLen) = $this->findAddressInTree($ipAddress);
        if( $pointer === 0 ) 
        {
            return [
                null, 
                $prefixLen
            ];
        }
        return [
            $this->resolveDataPointer($pointer), 
            $prefixLen
        ];
    }
    private function findAddressInTree($ipAddress)
    {
        $rawAddress = unpack('C*', inet_pton($ipAddress));
        $bitCount = count($rawAddress) * 8;
        $node = 0;
        $metadata = $this->metadata;
        if( $metadata->ipVersion === 6 ) 
        {
            if( $bitCount === 32 ) 
            {
                $node = $this->ipV4Start;
            }
        }
        else if( $metadata->ipVersion === 4 && $bitCount === 128 ) 
        {
            new InvalidArgum('Error looking up ' . $ipAddress . '. You attempted to look up an' . ' IPv6 address in an IPv4-only database.');
        }
        $nodeCount = $metadata->nodeCount;
        for( $i = 0; $i < $bitCount && $node < $nodeCount; ++$i ) 
        {
            $tempBit = 255 & $rawAddress[($i >> 3) + 1];
            $bit = 1 & $tempBit >> (7 - ($i % 8));
            $node = $this->readNode($node, $bit);
        }
        if( $node === $nodeCount ) 
        {
            return [
                0, 
                $i
            ];
        }
        else if( $nodeCount < $node ) 
        {
            return [
                $node, 
                $i
            ];
        }
        new InvalidDatabaseException('Something bad happened');
        return null;
    }
    private function ipV4StartNode()
    {
        if( $this->metadata->ipVersion === 4 ) 
        {
            return 0;
        }
        $node = 0;
        for( $i = 0; $i < 96 && $node < $this->metadata->nodeCount; ++$i ) 
        {
            $node = $this->readNode($node, 0);
        }
        return $node;
    }
    private function readNode($nodeNumber, $index)
    {
        $baseOffset = $nodeNumber * $this->metadata->nodeByteSize;
        switch( $this->metadata->recordSize ) 
        {
            case 24:
                $bytes = Util::read($this->fileHandle, $baseOffset + ($index * 3), 3);
                list(, $node) = unpack('N', '' . $bytes);
                return $node;
            case 28:
                $bytes = Util::read($this->fileHandle, $baseOffset + (3 * $index), 4);
                if( $index === 0 ) 
                {
                    $middle = (240 & ord($bytes[3])) >> 4;
                }
                else
                {
                    $middle = 15 & ord($bytes[0]);
                }
                list(, $node) = unpack('N', chr($middle) . substr($bytes, $index, 3));
                return $node;
            case 32:
                $bytes = Util::read($this->fileHandle, $baseOffset + ($index * 4), 4);
                list(, $node) = unpack('N', $bytes);
                return $node;
            default:
                new InvalidDatabaseException('Unknown record size: ' . $this->metadata->recordSize);
        }
    }
    private function resolveDataPointer($pointer)
    {
        $resolved = $pointer - $this->metadata->nodeCount + $this->metadata->searchTreeSize;
        if( $this->fileSize < $resolved ) 
        {
            new InvalidDatabaseException('The MaxMind DB file\'s search tree is corrupt');
        }
        list($data) = $this->decoder->decode($resolved);
        return $data;
    }
    private function findMetadataStart($filename)
    {
        $handle = $this->fileHandle;
        $fstat = fstat($handle);
        $fileSize = $fstat['size'];
        $marker = self::$METADATA_START_MARKER;
        $markerLength = self::$METADATA_START_MARKER_LENGTH;
        $metadataMaxLengthExcludingMarker = min(self::$METADATA_MAX_SIZE, $fileSize) - $markerLength;
        $i = 0;
        while( $i <= $metadataMaxLengthExcludingMarker ) 
        {
            $j = 0;
            while( $j < $markerLength ) 
            {
                fseek($handle, $fileSize - $i - $j - 1);
                $matchBit = fgetc($handle);
                if( $matchBit !== $marker[$markerLength - $j - 1] ) 
                {
                }
                else
                {
                    ++$j;
                }
                ++$i;
            }
            return $fileSize - $i;
        }
        new InvalidDatabaseException('Error opening database file (' . $filename . '). ' . 'Is this a valid MaxMind DB file?');
        return null;
    }
    public function metadata()
    {
        if( func_num_args() ) 
        {
            new InvalidArgum('Method takes no arguments.');
        }
        if( !is_resource($this->fileHandle) ) 
        {
            new InvalidArgum('Attempt to read from a closed MaxMind DB.');
        }
        return $this->metadata;
    }
    public function close()
    {
        if( !is_resource($this->fileHandle) ) 
        {
            new InvalidArgum('Attempt to close a closed MaxMind DB.');
        }
        fclose($this->fileHandle);
    }
}
class Metadata
{
    private $binaryFormatMajorVersion = null;
    private $binaryFormatMinorVersion = null;
    private $buildEpoch = null;
    private $databaseType = null;
    private $description = null;
    private $ipVersion = null;
    private $languages = null;
    private $nodeByteSize = null;
    private $nodeCount = null;
    private $recordSize = null;
    private $searchTreeSize = null;
    public function __construct($metadata)
    {
        $this->binaryFormatMajorVersion = $metadata['binary_format_major_version'];
        $this->binaryFormatMinorVersion = $metadata['binary_format_minor_version'];
        $this->buildEpoch = $metadata['build_epoch'];
        $this->databaseType = $metadata['database_type'];
        $this->languages = $metadata['languages'];
        $this->description = $metadata['description'];
        $this->ipVersion = $metadata['ip_version'];
        $this->nodeCount = $metadata['node_count'];
        $this->recordSize = $metadata['record_size'];
        $this->nodeByteSize = $this->recordSize / 4;
        $this->searchTreeSize = $this->nodeCount * $this->nodeByteSize;
    }
    public function __get($var)
    {
        return $this->$var;
    }
}
class Util
{
    public static function read($stream, $offset, $numberOfBytes)
    {
        if( $numberOfBytes === 0 ) 
        {
            return '';
        }
        if( fseek($stream, $offset) === 0 ) 
        {
            $value = fread($stream, $numberOfBytes);
            if( ftell($stream) - $offset === $numberOfBytes ) 
            {
                return $value;
            }
        }
        new InvalidDatabaseException('The MaxMind DB file contains bad data');
        return null;
    }
}
class Decoder
{
    private $fileStream = null;
    private $pointerBase = null;
    private $pointerBaseByteSize = null;
    private $pointerTestHack = null;
    private $switchByteOrder = null;
    const _EXTENDED = 0;
    const _POINTER = 1;
    const _UTF8_STRING = 2;
    const _DOUBLE = 3;
    const _BYTES = 4;
    const _UINT16 = 5;
    const _UINT32 = 6;
    const _MAP = 7;
    const _INT32 = 8;
    const _UINT64 = 9;
    const _UINT128 = 10;
    const _ARRAY = 11;
    const _CONTAINER = 12;
    const _END_MARKER = 13;
    const _BOOLEAN = 14;
    const _FLOAT = 15;
    public function __construct($fileStream, $pointerBase = 0, $pointerTestHack = false)
    {
        $this->fileStream = $fileStream;
        $this->pointerBase = $pointerBase;
        $this->pointerBaseByteSize = ($pointerBase > 0 ? log($pointerBase, 2) / 8 : 0);
        $this->pointerTestHack = $pointerTestHack;
        $this->switchByteOrder = $this->isPlatformLittleEndian();
    }
    public function decode($offset)
    {
        list(, $ctrlByte) = unpack('C', Util::read($this->fileStream, $offset, 1));
        ++$offset;
        $type = $ctrlByte >> 5;
        if( $type === 1 ) 
        {
            list($pointer, $offset) = $this->decodePointer($ctrlByte, $offset);
            if( $this->pointerTestHack ) 
            {
                return [$pointer];
            }
            list($result) = $this->decode($pointer);
            return [
                $result, 
                $offset
            ];
        }
        if( $type === 0 ) 
        {
            list(, $nextByte) = unpack('C', Util::read($this->fileStream, $offset, 1));
            $type = $nextByte + 7;
            if( $type < 8 ) 
            {
                new InvalidDatabaseException('Something went horribly wrong in the decoder. An extended type resolved to a type number < 8 (' . $type . ')');
                return null;
            }
            ++$offset;
        }
        list($size, $offset) = $this->sizeFromCtrlByte($ctrlByte, $offset);
        return $this->decodeByType($type, $offset, $size);
    }
    private function decodeByType($type, $offset, $size)
    {
        switch( $type ) 
        {
            case 7:
                return $this->decodeMap($size, $offset);
            case 11:
                return $this->decodeArray($size, $offset);
            case 14:
                return [
                    $this->decodeBoolean($size), 
                    $offset
                ];
            default:
                $newOffset = $offset + $size;
                $bytes = Util::read($this->fileStream, $offset, $size);
                switch( $type ) 
                {
                    case 4:
                    case 2:
                        return [
                            $bytes, 
                            $newOffset
                        ];
                    case 3:
                        $this->verifySize(8, $size);
                        return [
                            $this->decodeDouble($bytes), 
                            $newOffset
                        ];
                    case 15:
                        $this->verifySize(4, $size);
                        return [
                            $this->decodeFloat($bytes), 
                            $newOffset
                        ];
                    case 8:
                        return [
                            $this->decodeInt32($bytes, $size), 
                            $newOffset
                        ];
                    case 5:
                    case 6:
                    case 9:
                    case 10:
                        return [
                            $this->decodeUint($bytes, $size), 
                            $newOffset
                        ];
                    default:
                        new InvalidDatabaseException('Unknown or unexpected type: ' . $type);
                        return null;
                }
        }
    }
    private function verifySize($expected, $actual)
    {
        if( $expected !== $actual ) 
        {
            new InvalidDatabaseException('The MaxMind DB file\'s data section contains bad data (unknown data type or corrupt data)');
        }
    }
    private function decodeArray($size, $offset)
    {
        $array = [];
        for( $i = 0; $i < $size; ++$i ) 
        {
            list($value, $offset) = $this->decode($offset);
            array_push($array, $value);
        }
        return [
            $array, 
            $offset
        ];
    }
    private function decodeBoolean($size)
    {
        return ($size === 0 ? false : true);
    }
    private function decodeDouble($bits)
    {
        list(, $double) = unpack('d', $this->maybeSwitchByteOrder($bits));
        return $double;
    }
    private function decodeFloat($bits)
    {
        list(, $float) = unpack('f', $this->maybeSwitchByteOrder($bits));
        return $float;
    }
    private function decodeInt32($bytes, $size)
    {
        switch( $size ) 
        {
            case 0:
                return 0;
            case 1:
            case 2:
            case 3:
                $bytes = str_pad($bytes, 4, '', STR_PAD_LEFT);
                break;
            case 4:
                break;
            default:
                new InvalidDatabaseException('The MaxMind DB file\'s data section contains bad data (unknown data type or corrupt data)');
        }
        list(, $int) = unpack('l', $this->maybeSwitchByteOrder($bytes));
        return $int;
    }
    private function decodeMap($size, $offset)
    {
        $map = [];
        for( $i = 0; $i < $size; ++$i ) 
        {
            list($key, $offset) = $this->decode($offset);
            list($value, $offset) = $this->decode($offset);
            $map[$key] = $value;
        }
        return [
            $map, 
            $offset
        ];
    }
    private function decodePointer($ctrlByte, $offset)
    {
        $pointerSize = ($ctrlByte >> 3 & 3) + 1;
        $buffer = Util::read($this->fileStream, $offset, $pointerSize);
        $offset = $offset + $pointerSize;
        switch( $pointerSize ) 
        {
            case 1:
                $packed = pack('C', $ctrlByte & 7) . $buffer;
                list(, $pointer) = unpack('n', $packed);
                $pointer += $this->pointerBase;
                break;
            case 2:
                $packed = '' . pack('C', $ctrlByte & 7) . $buffer;
                list(, $pointer) = unpack('N', $packed);
                $pointer += ($this->pointerBase + 2048);
                break;
            case 3:
                $packed = pack('C', $ctrlByte & 7) . $buffer;
                list(, $pointer) = unpack('N', $packed);
                $pointer += ($this->pointerBase + 526336);
                break;
            case 4:
                $pointerOffset = $this->decodeUint($buffer, $pointerSize);
                $byteLength = $pointerSize + $this->pointerBaseByteSize;
                if( $byteLength <= _MM_MAX_INT_BYTES ) 
                {
                    $pointer = $pointerOffset + $this->pointerBase;
                    break;
                }
                else if( extension_loaded('gmp') ) 
                {
                    $pointer = _obf_0D1E16163B0417011133313F2732382C16231D230E1411(_obf_0D2A252E383D2932230A3E041C0C3C3B235B1D01063422($pointerOffset, $this->pointerBase));
                    break;
                }
                else if( extension_loaded('bcmath') ) 
                {
                    $pointer = bcadd($pointerOffset, $this->pointerBase);
                    break;
                }
                else
                {
                    new InvalidArgum('The gmp or bcmath extension must be installed to read this database.');
                    return null;
                }
        }
        return [
            $pointer, 
            $offset
        ];
    }
    private function decodeUint($bytes, $byteLength)
    {
        if( $byteLength === 0 ) 
        {
            return 0;
        }
        $integer = 0;
        for( $i = 0; $i < $byteLength; ++$i ) 
        {
            $part = ord($bytes[$i]);
            if( $byteLength <= _MM_MAX_INT_BYTES ) 
            {
                $integer = ($integer << 8) + $part;
            }
            else if( extension_loaded('gmp') ) 
            {
                $integer = _obf_0D1E16163B0417011133313F2732382C16231D230E1411(_obf_0D2A252E383D2932230A3E041C0C3C3B235B1D01063422(_obf_0D100E1205282809220501370906032D0F353F29050C11($integer, 256), $part));
            }
            else if( extension_loaded('bcmath') ) 
            {
                $integer = bcadd(bcmul($integer, 256), $part);
            }
            else
            {
                new InvalidArgum('The gmp or bcmath extension must be installed to read this database.');
                return null;
            }
        }
        return $integer;
    }
    private function sizeFromCtrlByte($ctrlByte, $offset)
    {
        $size = $ctrlByte & 31;
        if( $size < 29 ) 
        {
            return [
                $size, 
                $offset
            ];
        }
        $bytesToRead = $size - 28;
        $bytes = Util::read($this->fileStream, $offset, $bytesToRead);
        if( $size === 29 ) 
        {
            $size = 29 + ord($bytes);
        }
        else if( $size === 30 ) 
        {
            list(, $adjust) = unpack('n', $bytes);
            $size = 285 + $adjust;
        }
        else if( $size > 30 ) 
        {
            list(, $adjust) = unpack('N', '' . $bytes);
            $size = ($adjust & 268435455 >> (32 - (8 * $bytesToRead))) + 65821;
        }
        return [
            $size, 
            $offset + $bytesToRead
        ];
    }
    private function maybeSwitchByteOrder($bytes)
    {
        return ($this->switchByteOrder ? strrev($bytes) : $bytes);
    }
    private function isPlatformLittleEndian()
    {
        $testint = 255;
        $packed = pack('S', $testint);
        return $testint === current(unpack('v', $packed));
    }
}
